import java.util.*;

/**
 * A class that represents collections of customers.
 * The way this class is implemented at the moment is to store customer
 * objects in a Vector. (Vectors are used because they tie in
 * with Swing JLists.) Better would be if this class were implemented
 * to mediate between the program proper and a random-access file or
 * a database. That way data, and changes to the data, would be
 * persistent, i.e. would not be lost between runs of the program.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class CustomerCollection
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new, empty collection of customers.
    */
   public CustomerCollection()
   {  customers = new Vector();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns all customers in the collection.
    */
   public Vector findAll()
   {  return customers;
   }

   /**
    * Returns the customer in the collection with the specified 
    * identification number (if any). Although there can only be 
    * one such customer, a Vector is returned for consistency with
    * other search methods. It also means that with no match,
    * an empty Vector can be returned.
    *
    * @param theIdNum the identification number being sought.
    * @return the customer object with the specified id number or 
    * an empty Vector if there is no such customer
    */
   public Vector findById(String theIdNum)
   {  Vector matchList = new Vector();
      boolean found = false;
      Customer cust = null;
      Iterator iter = customers.iterator();
      while (iter.hasNext() && ! found)
      {  cust = (Customer) iter.next();
         found = cust.getIdNum().equals(theIdNum);
      }
      if (found)
      {  matchList.add(cust);
      }
      return matchList;
   }

   /**
    * Returns the customers in the collection whose names are equal to or
    * are similar to the specified name.
    *
    * @param theName the name being sought; must be non-null & non-empty.
    * @return a Vector of the customers in the colection whose
    * names match the specified name.
    */
   public Vector findByName(String theName)
   {  Vector matchList = new Vector();
      Iterator iter = customers.iterator();
      while (iter.hasNext())
      {  Customer cust = (Customer) iter.next();
         if (StringUtility.matches(cust.getName(), theName))
         {  matchList.add(cust);
         }
      }
      return matchList;
   }


/* --Setters----------------------------------------------------------- */

   /**
    * Add the specified customer object to the collection.
    *
    * @param theCust the customer being added to the collection; must be 
    * non-null & not already in the collection.
    */
   public void add(Customer theCust)
   {  customers.add(theCust);
   }

   /**
    * Remove the specified customer from the collection.
    *
    * @param theCustomer the customer to be removed; must be non-null and
    * a customer already in the collection.
    */
   public void remove(Customer theCustomer)
   {  customers.remove(theCustomer);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The customers in the collection.
    */
   private Vector customers;
}
