import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.*;

/**
 * A class that represents GUIs for inserting, modifying, deleting
 * and selecting customer orders.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class OrderGUI
   extends JDialog
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new Order GUI object for a specified collection
    * of orders.
    *
    * @param theParent the component that launched this GUI (a JFrame)
    * @param theOrderCollection the collection of orders.
    * @param theCDCollection the collection of CDs.
    */
   public OrderGUI(Frame theParent, 
      OrderCollection theOrderCollection, CDCollection theCDCollection)
   {  super(theParent, "CD-Direkt Orders", true);
      initialise(null, theOrderCollection, theCDCollection);
   }

   /**
    * Allocates a new Order GUI object for a specified customer's orders.
    *
    * @param theParent the component that launched this GUI (a JDialog)
    * @param theCust the customer whose orders we are to view.
    * @param theOrderCollection the collection of orders.
    * @param theCDCollection the collection of CDs.
    */
   public OrderGUI(Dialog theParent, Customer theCust,
      OrderCollection theOrderCollection, CDCollection theCDCollection)
   {  super(theParent, "CD-Direkt Orders", true);
      initialise(theCust, theOrderCollection, theCDCollection);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Setters----------------------------------------------------------- */

   /**
    * Displays the current order's details (if any) on the Order panel.
    */
   public void displayCurrentOrder()
   {  if (currentCustomer == null)
      {  custIdNumTF.setText("");
         custNameTF.setText("");
         custHouseStreetTF.setText("");
         custTownCountyTF.setText("");
         custTeleNumTF.setText("");
      }
      else
      {  custIdNumTF.setText(currentCustomer.getIdNum());
         custNameTF.setText(currentCustomer.getName());
         custHouseStreetTF.setText(currentCustomer.getHouseStreet());
         custTownCountyTF.setText(currentCustomer.getTownCounty());
         custTeleNumTF.setText(currentCustomer.getTeleNum());
      }
      if (currentOrder == null)
      {  idNumTF.setText("");
         dateTF.setText("");
         salesPersonTF.setText("");
         statusTF.setText("");
         recipientTF.setText("");
         shipHouseStreetTF.setText("");
         shipTownCountyTF.setText("");
         ccTypeTF.setText("");
         ccNumTF.setText("");
         ccNameTF.setText("");
         ccExpiryTF.setText("");
         linesList.setListData(new Vector());
      }
      else
      {  idNumTF.setText(currentOrder.getIdNum());
         dateTF.setText(currentOrder.getDate());
         salesPersonTF.setText(currentOrder.getSalesPerson());
         statusTF.setText(currentOrder.getStatus());
         recipientTF.setText(currentOrder.getRecipient());
         shipHouseStreetTF.setText(currentOrder.getShipHouseStreet());
         shipTownCountyTF.setText(currentOrder.getShipTownCounty());
         ccTypeTF.setText(currentOrder.getCCType());
         ccNumTF.setText(currentOrder.getCCNum());
         ccNameTF.setText(currentOrder.getCCName());
         ccExpiryTF.setText(currentOrder.getCCExpiry());
         linesList.setListData(currentOrder.getOrderLines());
      }
   }

   /**
    * If a CDGUI is returning to this OrderGUI, it can use this method
    * to say which CD the user has chosen.
    *
    * @param aCD the chosen CD.
    */
   public void setChosenCD(CD aCD)
   {  newCD = aCD;
      newCDIdNumTF.setText(newCD.getIdNum());
   }
      

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Initialises a new Order GUI object for a specified customer's orders.
    * (This is a helper method for two constructors.)
    *
    * @param theCust the customer whose orders we are to view.
    * @param theOrderCollection the collection of orders.
    * @param theCDCollection the collection of CDs.
    */
   private void initialise(Customer theCust, 
      OrderCollection theOrderCollection, CDCollection theCDCollection)
   {  currentCustomer = theCust;
      orderCollection = theOrderCollection;
      cdCollection = theCDCollection;
      JPanel searchPanel = createSearchPanel();
      JPanel orderPanel = createOrderPanel();
      JPanel returnPanel = createReturnPanel();
      Container contentPane = getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(searchPanel, BorderLayout.WEST);
      contentPane.add(orderPanel, BorderLayout.EAST);
      contentPane.add(returnPanel, BorderLayout.SOUTH);
      setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
      pack();
   }

   /**
    * Lays out a panel containing a list of current Orders.
    */
   private JPanel createSearchPanel()
   {  viewedOrders = orderCollection.findAll();
      orderList = new JList(viewedOrders);
      orderList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(orderList);

      orderList.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting()) // mouse wasn't yet at rest
            {  // do nothing
            }
            if (orderList.isSelectionEmpty()) // somehow, no selection was made
            {  // do nothing
            }
            else
            {  int index = orderList.getSelectedIndex(); 
               currentOrder = (Order) viewedOrders.get(index);
               currentCustomer = currentOrder.getCustomer();
               setGUIForOrderViewing();
            }
          }
      });

      JPanel p = new JPanel();
      p.add(scrollPane);
      return p;
   }

   /**
    * Lays out a panel containing a selected Order's details, and
    * offers update options.
    */
   private JPanel createOrderPanel()
   {  JLabel custIdNumLbl = new JLabel("Customer number:");
      custIdNumTF = new JTextField("", ID_TF_COLS);
      custIdNumTF.setEnabled(false);
      JLabel custNameLbl = new JLabel("Customer name:");
      custNameTF = new JTextField("", NAME_TF_COLS);
      custNameTF.setEnabled(false);
      JLabel custAddressLbl = new JLabel("Address:");
      custHouseStreetTF = new JTextField("", ADDRESS_TF_COLS);
      custHouseStreetTF.setEnabled(false);
      custTownCountyTF = new JTextField("", ADDRESS_TF_COLS);
      custTownCountyTF.setEnabled(false);
      JLabel custTeleNumLbl = new JLabel("Tel:");
      custTeleNumTF = new JTextField("", TELE_TF_COLS);
      custTeleNumTF.setEnabled(false);
      JLabel idNumLbl = new JLabel("Order number:");
      idNumTF = new JTextField("", ID_TF_COLS);
      idNumTF.setEnabled(false);
      JLabel dateLbl = new JLabel("Entry date:");
      dateTF = new JTextField("", DATE_TF_COLS);
      JLabel salesPersonLbl = new JLabel("Sales person:");
      salesPersonTF = new JTextField("", SALESPERSON_TF_COLS);
      JLabel statusLbl = new JLabel("Status:");
      statusTF = new JTextField("", STATUS_TF_COLS);
      JLabel recipientLbl = new JLabel("Ship to:");
      recipientTF = new JTextField("", NAME_TF_COLS);
      shipHouseStreetTF = new JTextField("", ADDRESS_TF_COLS);
      shipTownCountyTF = new JTextField("", ADDRESS_TF_COLS);
      JLabel ccTypeLbl = new JLabel("Credit card type:");
      ccTypeTF = new JTextField("", CCTYPE_TF_COLS);
      JLabel ccNumLbl = new JLabel("Credit card number:");
      ccNumTF = new JTextField("", CCNUM_TF_COLS);
      JLabel ccNameLbl = new JLabel("Credit card name:");
      ccNameTF = new JTextField("", NAME_TF_COLS);
      JLabel ccExpiryLbl = new JLabel("Credit card expiry:");
      ccExpiryTF = new JTextField("", DATE_TF_COLS);
      newBtn = new JButton(NEW);
      insertBtn = new JButton(INSERT);
      cancelBtn = new JButton(CANCEL);
      modifyBtn = new JButton(MODIFY);
      deleteBtn = new JButton(DELETE);
      JLabel linesLbl = new JLabel("Order lines:");
      linesList = new JList(new Vector());
      linesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(linesList);
      deleteLineBtn = new JButton(DELETE_LINE);
      cdSearchBtn = new JButton(FIND_CD);
      newCDIdNumTF = new JTextField("", CDID_TF_COLS);
      newCDIdNumTF.setEnabled(false);
      newCD = null;
      JLabel newQtyLbl = new JLabel("Quantity:");
      newQtyTF = new JTextField("", QTY_TF_COLS);
      saveLineBtn = new JButton(SAVE_LINE);
      setGUIForNoOrderSelected();

      newBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForOrderEntry();
         }
      });

      insertBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String salesPerson = salesPersonTF.getText().trim();
            String date = dateTF.getText().trim();
            String status = statusTF.getText().trim();
            String recipient = recipientTF.getText().trim();
            String shipHouseStreet = shipHouseStreetTF.getText().trim();
            String shipTownCounty = shipTownCountyTF.getText().trim();
            String ccType = ccTypeTF.getText().trim();
            String ccNum = ccNumTF.getText().trim();
            String ccName = ccNameTF.getText().trim();
            String ccExpiry = ccExpiryTF.getText().trim();
            if (date.equals("") || status.equals("") ||
                ccType.equals("") || ccNum.equals("") ||
                ccName.equals("") || ccExpiry.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "You must supply a date, salesperson, all credit card " +
                  "details and, optionally, a complete shipping address.", 
                  "Incomplete new order data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  if (recipient.equals(""))
               {  recipient = currentCustomer.getName();
               }
               if (shipHouseStreet.equals("") || shipTownCounty.equals(""))
               {  shipHouseStreet = currentCustomer.getHouseStreet();
                  shipTownCounty = currentCustomer.getTownCounty();
               }
               currentOrder = new Order(currentCustomer, salesPerson, date, 
                  recipient, shipHouseStreet, shipTownCounty, ccType,
                  ccNum, ccName, ccExpiry);
               orderCollection.add(currentOrder);
               setGUIForOrderViewing();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      });

      cancelBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForNoOrderSelected();
         }
      });

      modifyBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String date = dateTF.getText().trim();
            String status = statusTF.getText().trim();
            String recipient = recipientTF.getText().trim();
            String shipHouseStreet = shipHouseStreetTF.getText().trim();
            String shipTownCounty = shipTownCountyTF.getText().trim();
            String ccType = ccTypeTF.getText().trim();
            String ccNum = ccNumTF.getText().trim();
            String ccName = ccNameTF.getText().trim();
            String ccExpiry = ccExpiryTF.getText().trim();
            if (date.equals("") || status.equals("") ||
                ccType.equals("") || ccNum.equals("") ||
                ccName.equals("") || ccExpiry.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "You must supply a date, salesperson and all credit card " +
                  "details.",
                  "Incorrect changes made to order data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentOrder.setDate(date);
               currentOrder.setStatus(status);
               if (recipient.equals(""))
               {  recipient = currentCustomer.getName();
               }
               if (shipHouseStreet.equals("") || shipTownCounty.equals(""))
               {  shipHouseStreet = currentCustomer.getHouseStreet();
                  shipTownCounty = currentCustomer.getTownCounty();
               }
               currentOrder.setRecipient(recipient);
               currentOrder.setShipHouseStreet(shipHouseStreet);
               currentOrder.setShipTownCounty(shipTownCounty);
               currentOrder.setCCType(ccType);
               currentOrder.setCCNum(ccNum);
               currentOrder.setCCName(ccName);
               currentOrder.setCCExpiry(ccExpiry);
               setGUIForOrderViewing();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      });     

      deleteBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this Order. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
            if (confirm == JOptionPane.YES_OPTION)
            {  orderCollection.remove(currentOrder);
               currentCustomer.removeOrder(currentOrder);
               setGUIForNoOrderSelected();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      }); 

      deleteLineBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = linesList.getSelectedIndex();
            if (index == -1)
            {  JOptionPane.showMessageDialog(null,
                  "Select an order line from the list above before clicking " +
                  "on the Delete Line button.",
                  "No order line selected",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this order line. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
               if (confirm == JOptionPane.YES_OPTION)
               {  currentOrder.getOrderLines().removeElementAt(index);
                  setGUIForOrderViewing();
               }
            }
         }
      });

      cdSearchBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  CDGUI cdGUI = new CDGUI(OrderGUI.this, cdCollection);
            cdGUI.setVisible(true);
         }
      });

      saveLineBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String newQtyStr = newQtyTF.getText().trim();
            int newQty = 0;
            try
            {  newQty = Integer.parseInt(newQtyStr);
            }
            catch (NumberFormatException nfe)
            {  newQty = 0;
            }
            if (newCD == null || newQty <= 0) 
            {  JOptionPane.showMessageDialog(null,
                  "Choose a CD and specify a quantity greater than zero " +
                  "before clicking on the Save New Order Line button.",
                  "Invlaid new order line",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentOrder.addOrderLine(new OrderLine(newCD, newQty));
               setGUIForOrderViewing();
            }
         }
      });

      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(custIdNumLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custIdNumTF, p, gbl, gbc, 
         0, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custNameLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custNameTF, p, gbl, gbc, 
         1, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custAddressLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custHouseStreetTF, p, gbl, gbc, 
         2, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custTownCountyTF, p, gbl, gbc, 3, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 3, 1, 1, 0, 0);
      GuiUtility.addComp(custTeleNumLbl, p, gbl, gbc, 3, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 5, 1, 1, 0, 0);
      GuiUtility.addComp(custTeleNumTF, p, gbl, gbc, 
         3, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 4, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 1, 1, 1, 0, 0);
      GuiUtility.addComp(idNumTF, p, gbl, gbc, 
         4, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 3, 1, 1, 0, 0);
      GuiUtility.addComp(salesPersonLbl, p, gbl, gbc, 4, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 5, 1, 1, 0, 0);
      GuiUtility.addComp(salesPersonTF, p, gbl, gbc, 
         4, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(dateLbl, p, gbl, gbc, 5, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 1, 1, 1, 0, 0);
      GuiUtility.addComp(dateTF, p, gbl, gbc, 
         5, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 3, 1, 1, 0, 0);
      GuiUtility.addComp(statusLbl, p, gbl, gbc, 5, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 5, 1, 1, 0, 0);
      GuiUtility.addComp(statusTF, p, gbl, gbc, 
         5, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(recipientLbl, p, gbl, gbc, 6, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         6, 1, 1, 1, 0, 0);
      GuiUtility.addComp(recipientTF, p, gbl, gbc, 
         6, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(shipHouseStreetTF, p, gbl, gbc, 
         7, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(shipTownCountyTF, p, gbl, gbc, 
         8, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(ccTypeLbl, p, gbl, gbc, 9, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 1, 1, 1, 0, 0);
      GuiUtility.addComp(ccTypeTF, p, gbl, gbc, 
         9, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 3, 1, 1, 0, 0);
      GuiUtility.addComp(ccNumLbl, p, gbl, gbc, 9, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 5, 1, 1, 0, 0);
      GuiUtility.addComp(ccNumTF, p, gbl, gbc, 
         9, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(ccNameLbl, p, gbl, gbc, 10, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 1, 1, 1, 0, 0);
      GuiUtility.addComp(ccNameTF, p, gbl, gbc, 
         10, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 3, 1, 1, 0, 0);
      GuiUtility.addComp(ccExpiryLbl, p, gbl, gbc, 10, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 5, 1, 1, 0, 0);
      GuiUtility.addComp(ccExpiryTF, p, gbl, gbc, 
         10, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl1 = new JPanel();
      btnPnl1.setLayout(new BoxLayout(btnPnl1, BoxLayout.X_AXIS));
      btnPnl1.add(newBtn);
      btnPnl1.add(insertBtn);
      btnPnl1.add(cancelBtn);
      JPanel btnPnl2 = new JPanel();
      btnPnl2.setLayout(new BoxLayout(btnPnl2, BoxLayout.X_AXIS));
      btnPnl2.add(modifyBtn);
      btnPnl2.add(deleteBtn);
      GuiUtility.addComp(btnPnl1, p, gbl, gbc, 11, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl2, p, gbl, gbc, 12, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.NORTHWEST;
      GuiUtility.addComp(linesLbl, p, gbl, gbc, 13, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         13, 1, 1, 1, 0, 0);
      GuiUtility.addComp(scrollPane, p, gbl, gbc, 
         13, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl3 = new JPanel();
      btnPnl3.setLayout(new BoxLayout(btnPnl3, BoxLayout.X_AXIS));
      btnPnl3.add(deleteLineBtn);
      GuiUtility.addComp(btnPnl3, p, gbl, gbc, 14, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      JPanel btnPnl4 = new JPanel();
      GridBagLayout gblP = new GridBagLayout();
      GridBagConstraints gbcP = new GridBagConstraints();
      gbcP.fill = GridBagConstraints.NONE;
      gbcP.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(cdSearchBtn, btnPnl4, gblP, gbcP, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(newCDIdNumTF, btnPnl4, gblP, gbcP,
         0, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 3, 1, 1, 0, 0);
      GuiUtility.addComp(newQtyLbl, btnPnl4, gblP, gbcP, 0, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 5, 1, 1, 0, 0);
      GuiUtility.addComp(newQtyTF, btnPnl4, gblP, gbcP, 
         0, 6, 1, GridBagConstraints.REMAINDER, 0, 0); 
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 7, 1, 1, 0, 0);
      GuiUtility.addComp(saveLineBtn, btnPnl4, gblP, gbcP, 
         0, 8, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl4, p, gbl, gbc, 15, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Creates the button that takes the user back to the parent
    * component, i.e. the frame from which s/he came before viewing
    * this frame.
    */
   private JPanel createReturnPanel()
   {  returnBtn = new JButton(RETURN);
      returnBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setVisible(false);
            dispose();
         }
      });
      JPanel p = new JPanel();
      p.add(returnBtn);
      return p;
   }

   /**
    * Disables/enables parts of the GUI for when the user has no
    * customer/order selected.
    */
   private void setGUIForNoOrderSelected()
   {  if (currentCustomer == null)
      {  newBtn.setEnabled(false);
      }
      else
      {  newBtn.setEnabled(true);
      }
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      orderList.setEnabled(true);
      currentOrder = null;
      displayCurrentOrder();
      salesPersonTF.setEnabled(false);
      dateTF.setEnabled(false);
      statusTF.setEnabled(false);
      recipientTF.setEnabled(false);
      shipHouseStreetTF.setEnabled(false);
      shipTownCountyTF.setEnabled(false);
      ccTypeTF.setEnabled(false);
      ccNumTF.setEnabled(false);
      ccNameTF.setEnabled(false);
      ccExpiryTF.setEnabled(false);
      linesList.setEnabled(false);
      deleteLineBtn.setEnabled(false);
      cdSearchBtn.setEnabled(false);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText("");
      newQtyTF.setEnabled(false);
      saveLineBtn.setEnabled(false);
   }

   /**
    * Disables/enables parts of the GUI for when the user is viewing
    * an Order.
    */
   private void setGUIForOrderViewing()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(true);
      deleteBtn.setEnabled(true);
      orderList.setEnabled(true);
      displayCurrentOrder();
      salesPersonTF.setEnabled(true);
      dateTF.setEnabled(true);
      statusTF.setEnabled(true);
      recipientTF.setEnabled(true);
      shipHouseStreetTF.setEnabled(true);
      shipTownCountyTF.setEnabled(true);
      ccTypeTF.setEnabled(true);
      ccNumTF.setEnabled(true);
      ccNameTF.setEnabled(true);
      ccExpiryTF.setEnabled(true);
      linesList.setEnabled(true);
      deleteLineBtn.setEnabled(true);
      cdSearchBtn.setEnabled(true);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText("");
      newQtyTF.setEnabled(true);
      saveLineBtn.setEnabled(true);
   }

   /**
    * Disables/enables parts of the GUI for when the user is entering
    * a new Order.
    */
   private void setGUIForOrderEntry()
   {  newBtn.setEnabled(false);
      insertBtn.setEnabled(true);
      cancelBtn.setEnabled(true);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      orderList.setEnabled(false);
      currentOrder = null;
      displayCurrentOrder();
      salesPersonTF.setEnabled(true);
      dateTF.setEnabled(true);
      statusTF.setEnabled(true);
      recipientTF.setEnabled(true);
      shipHouseStreetTF.setEnabled(true);
      shipTownCountyTF.setEnabled(true);
      ccTypeTF.setEnabled(true);
      ccNumTF.setEnabled(true);
      ccNameTF.setEnabled(true);
      ccExpiryTF.setEnabled(true);
      linesList.setEnabled(false);
      deleteLineBtn.setEnabled(false);
      cdSearchBtn.setEnabled(false);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText(""); 
      newQtyTF.setEnabled(false);
      saveLineBtn.setEnabled(false);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The order collection that this GUI presents to the users.
    * This may be all orders known to the system or, when this GUI
    * is launched from a CustomerGUI, it will be orders for one
    * particular customer.
    */
   private OrderCollection orderCollection;

   /**
    * The CD collection that this GUI presents to the users.
    * Supplied in this class' constructor, and used when launching
    * a CD GUI.)
    */
   private CDCollection cdCollection;

   /**
    * The order whose details are currently being displayed, if any.
    */
   private Order currentOrder;

   /**
    * The customer for the current order.
    */
   private Customer currentCustomer;

   /**
    * A subset of the order collection that is currently being listed
    * on the screen.
    */
   private Vector viewedOrders;

   /**
    * On return from a CDGUI, allowing a user to choose a CD to add to
    * an order, this variable will hold that chosen CD.
    */
   private CD newCD;

   /**
    * A JLIst that displays a list of all the orders in the
    * order collection, specifically those in viewedOrders.
    */
   private JList orderList;

   /**
    * A textfield for displaying the identification number of the 
    * customer who placed the order.
    */
   private JTextField custIdNumTF;


   /**
    * A textfield for displaying the name of the customer who placed
    * the order.
    */
   private JTextField custNameTF;

   /**
    * A textfield for displaying the address (house number and street
    * name) of the customer who placed the order.
    */
   private JTextField custHouseStreetTF;

   /**
    * A textfield for displaying the address (town and country) of the 
    * customer who placed the order.
    */
   private JTextField custTownCountyTF;

   /**
    * A textfield for displaying the telephone number of the customer
    * who placed the order.
    */
   private JTextField custTeleNumTF;

   /**
    * A textfield for displaying the identification number of the current 
    * order.
    */
   private JTextField idNumTF;

   /**
    * A textfield for displaying the name of the salesperson who took
    * the order.
    */
   private JTextField salesPersonTF;

   /**
    * A textfield for displaying the date the order was placed.
    */
   private JTextField dateTF;

   /**
    * A textfield for displaying the status of the current order.
    */
   private JTextField statusTF;

   /**
    * A textfield for displaying the name of the person to whom the
    * order will be shipped.
    */
   private JTextField recipientTF;

   /**
    * A textfield for displaying the address (house number and street
    * name) of the order recipient.
    */
   private JTextField shipHouseStreetTF;

   /**
    * A textfield for displaying the address (town and country) of the 
    * order recipient.
    */
   private JTextField shipTownCountyTF;

   /**
    * A textfield for displaying the type of credit card used for payment.
    */
   private JTextField ccTypeTF;

   /**
    * A textfield for displaying the credit card number.
    */
   private JTextField ccNumTF;

   /**
    * A textfield for displaying the name on the credit card.
    */
   private JTextField ccNameTF;

   /**
    * A textfield for displaying the credit card expiry date.
    */
   private JTextField ccExpiryTF;

   /**
    * A button for requesting that the screen be set up for insertion of
    * details of a new order.
    */
   private JButton newBtn;

   /**
    * A button for indicating that entry of the details of a new order is
    * complete, the details should be taken from the screen and a new
    * order created and added to the collection.
    */
   private JButton insertBtn;

   /**
    * A button for aborting entry of a new order.
    */
   private JButton cancelBtn;

   /**
    * A button for indicating that changes made to the details of the
    * current order should be taken from the screen and used to update the
    * object in memory.
    */
   private JButton modifyBtn;

   /**
    * A button for requesting deletion of the current order.
    */
   private JButton deleteBtn;

   /**
    * A JList for display of the order lines on the current order.
    */
   private JList linesList;

   /**
    * A button for requesting that the line selected in the linesList
    * JLIst be deleted.
    */
   private JButton deleteLineBtn;

   /**
    * A button for launching a CDGUI so that the user can choose a 
    * CD to add to the order.
    */
   private JButton cdSearchBtn;

   /**
    * A textfield for display of the chosen CD's identification number
    * when entering a new order line.
    */
   private JTextField newCDIdNumTF;

   /**
    * A textfield for entry of the quantity of the chosen CD being ordered
    * on a new order line.
    */
   private JTextField newQtyTF;

   /**
    * A button for indicating that entry of a new order line is complete,
    * details should be taken from the screen and used to update the
    * current order in memory.
    */
   private JButton saveLineBtn;

   /**
    * A button for requesting termination of this dialog, and return to
    * the parent JFrame or JDialog.
    */
   private JButton returnBtn;

   /**
    * The width of the identification number textfields.
    */
   private static final int ID_TF_COLS = 6;

   /**
    * The width of the customer name textfields.
    */
   private static final int NAME_TF_COLS = 30;

   /**
    * The width of the address textfields.
    */
   private static final int ADDRESS_TF_COLS = 30;

   /**
    * The width of the telephone number textfield.
    */
   private static final int TELE_TF_COLS = 10;

   /**
    * The width of the date textfields.
    */
   private static final int DATE_TF_COLS = 8;

   /**
    * The width of the recipient name textfield.
    */
   private static final int SALESPERSON_TF_COLS = 20;

   /**
    * The width of the order status textfield.
    */
   private static final int STATUS_TF_COLS = 20;

   /**
    * The width of the credit card type textfield.
    */
   private static final int CCTYPE_TF_COLS = 10;

   /**
    * The width of the credit card number textfield.
    */
   private static final int CCNUM_TF_COLS = 20;

   /**
    * The width of the new CD identification number textfield.
    */
   private static final int CDID_TF_COLS = 10;

   /**
    * The width of the order line quantitytextfield.
    */
   private static final int QTY_TF_COLS = 6;

   /**
    * The text on newBtn.
    */
   private static final String NEW = "NEW ORDER";

   /**
    * The text on insertBtn.
    */
   private static final String INSERT = "INSERT";

   /** 
    * The text on cancelBtn.
    */
   private static final String CANCEL = "CANCEL";

   /**
    * The text on modifyBtn.
    */
   private static final String MODIFY = "Save Changes";

   /**
    * The text on deleteBtn.
    */
   private static final String DELETE = "DELETE";

   /**
    * The text on deleteLineBtn.
    */
   private static final String DELETE_LINE = "Delete Line";

   /**
    * The text on cdSearchBtn.
    */
   private static final String FIND_CD = "Choose CD";

   /**
    * The text on saveLineBtn.
    */
   private static final String SAVE_LINE = "Save New Order Line";

   /**
    * The text on returnBtn.
    */
   private static final String RETURN = "RETURN";
}
