import java.util.*;

/**
 * A class that represents customer order objects.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class Order
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new order object for the given customer.
    *
    * @param theCustomer the customer who has placed this order; must be
    * non-null.
    * @param theSalesPerson the sales person who took this order; must be
    * non-null & non-empty.
    * @param theDate the date the order was placed; must be non-null,
    * non-empty & a legitimate date.
    * @param theRecipient the name of the person to whom we ship; must be
    * non-null; can be empty (when we ship to customer).
    * @param theHouseStreet the house/street to which we ship; must be
    * non-null; can be empty.
    * @param theTownCounty the town/county to which we ship; must be
    * non-null; can be empty.
    * @param theCCType the type of credit card; must be non-null, non-empty
    * & one of the legal credit card types.
    * @param theCCNum the credit card number; must be non-null, non-empty
    * & a legal card number.
    * @param theCCName the credit card name; must be non-null & non-empty.
    * @param theCCExpiry the credit card expiry date; must be non-null,
    * non-empty & a legitimate date in the future.
    */
   public Order(Customer theCustomer, String theSalesPerson,
      String theDate, String theRecipient, String theHouseStreet,
      String theTownCounty, String theCCType, String theCCNum, 
      String theCCName, String theCCExpiry)
   {  idNum = PREFIX + nextIdNum;
      nextIdNum++;
      customer = theCustomer;
      salesPerson = theSalesPerson;
      date = theDate;
      recipient = theRecipient;
      shipHouseStreet = theHouseStreet;
      shipTownCounty = theTownCounty;
      ccType = theCCType;
      ccNum = theCCNum;
      ccName = theCCName;
      ccExpiry = theCCExpiry;
      orderLines = new Vector();
      status = NEW;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns the order's unique identification number
    *
    * @return the identification number.
    */
   public String getIdNum()
   {  return idNum;
   }

   /**
    * Returns the order's customer.
    *
    * @return the customer.
    */
   public Customer getCustomer()
   {  return customer;
   }

   /**
    * Returns the order's sales person.
    *
    * @return the salesperson.
    */
   public String getSalesPerson()
   {  return salesPerson;
   }

   /**
    * Returns the date the order was taken.
    *
    * @return the date the order was taken.
    */
   public String getDate()
   {  return date;
   }

   /**
    * Returns the order recipient, to whom we ship.
    *
    * @return the recipient.
    */
   public String getRecipient()
   {  return recipient;
   }

   /**
    * Returns the recipient's house/street.
    *
    * @return the recipient's house street.
    */
   public String getShipHouseStreet()
   {  return shipHouseStreet;
   }

   /**
    * Returns the recipient's town/county.
    *
    * @return the recipient's town/county.
    */
   public String getShipTownCounty()
   {  return shipTownCounty;
   }

   /**
    * Returns the credit card type (Visa/AmEx, etc.)
    *
    * @return the credit card type.
    */
   public String getCCType()
   {  return ccType;
   }

   /**
    * Returns the credit card number.
    *
    * @return the credit card numver.
    */
   public String getCCNum()
   {  return ccNum;
   }

   /**
    * Returns the name, spelled as it is on the credit card.
    *
    * @return the name on the credit card.
    */
   public String getCCName()
   {  return ccName;
   }

   /**
    * Returns the credit card's expiry date.
    *
    * @return the credit card expiry date.
    */
   public String getCCExpiry()
   {  return ccExpiry;
   }

   /**
    * Returns the order lines from the order.
    *
    * @return the order lines.
    */
   public Vector getOrderLines()
   {  return orderLines;
   }

   /**
    * Returns the order's status (new, waiting for stock, shipped).
    *
    * @return the order's status.
    */
   public String getStatus()
   {  return status;
   }


   /**
    * Returns the order's total price.
    *
    * @return the order's total price.
    */
   public int getTotalPrice()
   {  int totalPrice = 0;
      Iterator iter = orderLines.iterator();
      while (iter.hasNext())
      {  totalPrice += ((OrderLine) iter.next()).getPrice();
      }
      return totalPrice;
   }
   
/* --Setters----------------------------------------------------------- */

   /**
    * Sets the order's customer to the specified customer.
    *
    * @param theCustomer the revised customer; must be non-null.
    */
   public void setCustomer(Customer theCustomer)
   {  customer = theCustomer;
   }

   /**
    * Sets the order's sales person to the specified sales person.
    *
    * @param theSalesPerson the revised salesperson; must be non-null and 
    * non-empty.
    */
   public void setSalesPerson(String theSalesPerson)
   {  salesPerson = theSalesPerson;
   }

   /**
    * Sets the date the order was taken to the specified date.
    *
    * @param theDate the revised date the order was taken; must be non-null,
    * non-empty and a legitimate date.
    */
   public void setDate(String theDate)
   {  date = theDate;
   }

   /**
    * Sets the order recipient to the specified value.
    *
    * @param theRecipient the revised recipient; must be non-null; can
    * be empty (when we ship to customer).
    */
   public void setRecipient(String theRecipient)
   {  recipient = theRecipient;
   }

   /**
    * Sets the recipient's house/street to the specified value.
    *
    * @param theShipHouseStreet the revised recipient's house/street;
    * must be non-null; can be empty.
    */
   public void setShipHouseStreet(String theShipHouseStreet)
   {  shipHouseStreet = theShipHouseStreet;
   }

   /**
    * Sets the recipient's town/county to the specified value.
    *
    * @param theShipTownCounty the revised recipient's town/county;
    * must be non-null; can be empty.
    */
   public void setShipTownCounty(String theShipTownCounty)
   {  shipTownCounty = theShipTownCounty;
   }

   /**
    * Sets the credit card type to the specified value.
    *
    * @param theCCType the revised credit card type; must be non-null,
    * non-empty and one of the recognised card types.
    */
   public void setCCType(String theCCType)
   {  ccType = theCCType;
   }

   /**
    * Sets the credit card number to the specified value.
    *
    * @param theCCNum the revised credit card number; must be non-null,
    * non-empty and a legal card number.
    */
   public void setCCNum(String theCCNum)
   {  ccNum = theCCNum;
   }

   /**
    * Sets the credit card holder's name to the specified value.
    *
    * @param theCCName the revised name on the credit card; must be
    * non-null & non-empty.
    */
   public void setCCName(String theCCName)
   {  ccName = theCCName;
   }

   /**
    * Sets the credit card's expiry date to the specified value.
    *
    * @param theCCExpiry the revised credit card expiry date; must be
    * non-null, non-empty & a legitimate date in the future.
    */
   public void setCCExpiry(String theCCExpiry)
   {  ccExpiry = theCCExpiry;
   }

   /**
    * Adds an order line to the order.
    *
    * @param theOrderLine the new order line; must be non-null.
    */
   public void addOrderLine(OrderLine theOrderLine)
   {  orderLines.add(theOrderLine);
   }

   /**
    * Remove the specified order line from the order.
    *
    * @param theOrderLine the order line to be removed; must be non-null 
    * & must be an order line that is in the order's collection of order 
    * lines.
    */
   public void removeOrderLine(OrderLine theOrderLine)
   {  orderLines.remove(theOrderLine);
   }

   /**
    * Sets the order's status to the specified value.
    *
    * @param theStatus the new status; must be one of NEW, WAITING, SHIPPED.
    */
   public void setStatus(String theStatus)
   {  status = theStatus;
   }

/* --Common object interface------------------------------------------- */

   /**
    * Returns a string representation of parts of this order.
    *
    * @return a string representation of this order.
    */
   public String toString()
   {  return idNum + " " + date + " " + status;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   /**
    * The order's unique identification code (a prefix and a number).
    */
   private String idNum;

   /**
    * The customer who has placed this order.
    */
   private Customer customer;

   /**
    * The name of the sales person who took this order.
    */
   private String salesPerson;

   /**
    * The date the order was placed.
    */
   private String date;

   /**
    * The name of the person to whom we ship.
    */
   private String recipient;

   /**
    * The address (house number and street name) to which we ship.
    */
   private String shipHouseStreet;

   /**
    * The address (town and county) to which we ship.
    */
   private String shipTownCounty;

   /**
    * The type of credit card.
    */
   private String ccType;

   /**
    * The credit card number.
    */
   private String ccNum;

   /**
    * The name on the credit card.
    */
   private String ccName;

   /**
    * The credit card expiry date.
    */
   private String ccExpiry;

   /**
    * The order status (e.g. new, waiting for stock, shipped)
    */
   private String status;

   /**
    * The set of order lines on this order.
    */
   private Vector orderLines;

   /**
    * The identification number to use for the next new order.
    */
   private static int nextIdNum = 8000;

   /**
    * Identification codes comprise this prefix and a unique integer.
    */
   private static final String PREFIX = "ORD";

   /**
    * Order status: a newly recorded order.
    */
   private static final String NEW = "NEW";

   /**
    * Order status: an order that cannot be shipped until stock has 
    * arrived.
    */
   private static final String WAITING = "WAITING FOR STOCK";

   /**
    * Order status: an order that has been shipped.
    */
   private static final String SHIPPED = "SHIPPED";
}
