import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.*;

/**
 * A class that represents GUIs for inserting, modifying, deleting
 * and selecting customers from a customer collection.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class CustomerGUI
   extends JDialog
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new customer GUI object for a specified collection of
    * customers.
    *
    * @param theParent the component that launched this GUI.
    * @param theCustCollection the collection of customers.
    * @param theCDCollection the collection of CDs.
    */
   public CustomerGUI(Frame theParent, 
      CustomerCollection theCustCollection, CDCollection theCDCollection)
   {  super(theParent, "CD-Direkt Customers", true);
      custCollection = theCustCollection;
      cdCollection = theCDCollection;
      JPanel searchPanel = createSearchPanel();
      JPanel customerPanel = createCustomerPanel();
      JPanel returnPanel = createReturnPanel();
      Container contentPane = getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(searchPanel, BorderLayout.WEST);
      contentPane.add(customerPanel, BorderLayout.EAST);
      contentPane.add(returnPanel, BorderLayout.SOUTH);
      setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
      pack();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Setters----------------------------------------------------------- */

   /**
    * Displays the current customer's details (if any) on the
    * customer panel.
    */
   public void displayCurrentCustomer()
   {  if (currentCustomer == null)
      {  idNumTF.setText("");
         nameTF.setText("");
         houseStreetTF.setText("");
         townCountyTF.setText("");
         teleNumTF.setText("");
      }
      else
      {  idNumTF.setText(currentCustomer.getIdNum());
         nameTF.setText(currentCustomer.getName());
         houseStreetTF.setText(currentCustomer.getHouseStreet());
         townCountyTF.setText(currentCustomer.getTownCounty());
         teleNumTF.setText(currentCustomer.getTeleNum());
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Lays out a panel containing a list of current customers, and
    * offers search options.
    */
   private JPanel createSearchPanel()
   {  viewedCustomers = custCollection.findAll();
      custList = new JList(viewedCustomers);
      custList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(custList);
      JLabel idNumLbl = new JLabel("Customer number:");
      searchIdNumTF = new JTextField("", ID_TF_COLS);
      JLabel nameLbl = new JLabel("Customer name:");
      searchNameTF = new JTextField("", NAME_TF_COLS);
      searchBtn = new JButton(SEARCH);
      fullBtn = new JButton(FULL);

      custList.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting()) // mouse wasn't yet at rest
            {  // do nothing
            }
            if (custList.isSelectionEmpty()) // somehow, no selection was made
            {  // do nothing
            }
            else
            {  int index = custList.getSelectedIndex(); 
               currentCustomer = (Customer) viewedCustomers.get(index);
               setGUIForCustomerViewing();
            }
          }
      });

      searchBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String idNum = searchIdNumTF.getText().trim();
            String name = searchNameTF.getText().trim();
            if (! idNum.equals("") && name.equals("")) 
            {  viewedCustomers = custCollection.findById(idNum);
               custList.setListData(viewedCustomers);
               custList.setSelectedIndex(-1);
            }
            else if (idNum.equals("") && ! name.equals(""))
            {  viewedCustomers = custCollection.findByName(name);
               custList.setListData(viewedCustomers);
               custList.setSelectedIndex(-1);
            }
            else
            {  JOptionPane.showMessageDialog(null,
                  "To search, you must supply either a customer " +
                  "identification number or a customer name.",
                  "Error in search values",
                  JOptionPane.ERROR_MESSAGE);
            }
         }
      });

      fullBtnLstnr = new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  searchIdNumTF.setText("");
            searchNameTF.setText("");
            viewedCustomers = custCollection.findAll();
            custList.setListData(viewedCustomers);
            custList.setSelectedIndex(-1);
         }
      };
      fullBtn.addActionListener(fullBtnLstnr);

      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      GuiUtility.addComp(scrollPane, p, gbl, gbc, 0, 0, 
         1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(searchIdNumTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
      GuiUtility.addComp(nameLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(searchNameTF, p, gbl, gbc, 2, 2, 1, 1, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      GuiUtility.addComp(searchBtn, p, gbl, gbc, 3, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(fullBtn, p, gbl, gbc, 4, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Lays out a panel containing a selected customer's details, and
    * offers update options.
    */
   private JPanel createCustomerPanel()
   {  JLabel idNumLbl = new JLabel("Customer number:");
      idNumTF = new JTextField("", ID_TF_COLS);
      idNumTF.setEnabled(false);
      JLabel nameLbl = new JLabel("Customer name:");
      nameTF = new JTextField("", NAME_TF_COLS);
      JLabel addressLbl = new JLabel("Address:");
      houseStreetTF = new JTextField("", ADDRESS_TF_COLS);
      townCountyTF = new JTextField("", ADDRESS_TF_COLS);
      JLabel teleNumLbl = new JLabel("Tel:");
      teleNumTF = new JTextField("", TELE_TF_COLS);
      newBtn = new JButton(NEW);
      insertBtn = new JButton(INSERT);
      cancelBtn = new JButton(CANCEL);
      modifyBtn = new JButton(MODIFY);
      deleteBtn = new JButton(DELETE);
      ordersBtn = new JButton(ORDERS);
      setGUIForNoCustomerSelected();

      newBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForCustomerEntry();
         }
      });

      insertBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String name = nameTF.getText().trim();
            String houseStreet = houseStreetTF.getText().trim();
            String townCounty = townCountyTF.getText().trim();
            String teleNum = teleNumTF.getText().trim();
            if (name.equals("") || houseStreet.equals("") ||
                townCounty.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "You must supply a customer name and a full address.",
                  "Incomplete new customer data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentCustomer = 
                  new Customer(name, houseStreet, townCounty, teleNum);
               custCollection.add(currentCustomer);
               setGUIForCustomerViewing();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      });

      cancelBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForNoCustomerSelected();
         }
      });

      modifyBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String name = nameTF.getText().trim();
            String houseStreet = houseStreetTF.getText().trim();
            String townCounty = townCountyTF.getText().trim();
            String teleNum = teleNumTF.getText().trim();
            if (name.equals("") || houseStreet.equals("") ||
                townCounty.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "The customer name and address cannot be blank.",
                  "Trying to make fields blank",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentCustomer.setName(name);
               currentCustomer.setHouseStreet(houseStreet);
               currentCustomer.setTownCounty(townCounty);
               currentCustomer.setTeleNum(teleNum);
               setGUIForCustomerViewing();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      });     

      deleteBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this customer. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
            if (confirm == JOptionPane.YES_OPTION)
            {  custCollection.remove(currentCustomer);
               setGUIForNoCustomerSelected();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      }); 

      ordersBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  OrderGUI orderGUI = new OrderGUI(
               CustomerGUI.this, currentCustomer, new OrderCollection(
               currentCustomer.getCurrentOrders()), cdCollection);
            orderGUI.setVisible(true);
         }
      });
               
      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(idNumTF, p, gbl, gbc, 
         0, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(nameLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(nameTF, p, gbl, gbc, 
         1, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(addressLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(houseStreetTF, p, gbl, gbc, 
         2, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(townCountyTF, p, gbl, gbc, 3, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 3, 1, 1, 0, 0);
      GuiUtility.addComp(teleNumLbl, p, gbl, gbc, 3, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 5, 1, 1, 0, 0);
      GuiUtility.addComp(teleNumTF, p, gbl, gbc, 
         3, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl1 = new JPanel();
      btnPnl1.setLayout(new BoxLayout(btnPnl1, BoxLayout.X_AXIS));
      btnPnl1.add(newBtn);
      btnPnl1.add(insertBtn);
      btnPnl1.add(cancelBtn);
      JPanel btnPnl2 = new JPanel();
      btnPnl2.setLayout(new BoxLayout(btnPnl2, BoxLayout.X_AXIS));
      btnPnl2.add(modifyBtn);
      btnPnl2.add(deleteBtn);
      JPanel btnPnl3 = new JPanel();
      btnPnl3.setLayout(new BoxLayout(btnPnl3, BoxLayout.X_AXIS));
      btnPnl3.add(ordersBtn);
      GuiUtility.addComp(btnPnl1, p, gbl, gbc, 4, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl2, p, gbl, gbc, 5, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl3, p, gbl, gbc, 6, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Creates the button that takes the user back to the parent
    * component, i.e. the frame from which s/he came before viewing
    * this frame.
    */
   private JPanel createReturnPanel()
   {  returnBtn = new JButton(RETURN);
      returnBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setVisible(false);
            dispose();
         }
      });
      JPanel p = new JPanel();
      p.add(returnBtn);
      return p;
   }

   /**
    * Disables/enables parts of the GUI for when the user has no
    * customer selected.
    */
   private void setGUIForNoCustomerSelected()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      ordersBtn.setEnabled(false);
      custList.setEnabled(true);
      currentCustomer = null;
      displayCurrentCustomer();
      nameTF.setEnabled(false);
      houseStreetTF.setEnabled(false);
      townCountyTF.setEnabled(false);
      teleNumTF.setEnabled(false);
   }

   /**
    * Disables/enables parts of the GUI for when the user is viewing
    * a customer.
    */
   private void setGUIForCustomerViewing()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(true);
      deleteBtn.setEnabled(true);
      ordersBtn.setEnabled(true);
      custList.setEnabled(true);
      displayCurrentCustomer();
      nameTF.setEnabled(true);
      houseStreetTF.setEnabled(true);
      townCountyTF.setEnabled(true);
      teleNumTF.setEnabled(true);
   }

   /**
    * Disables/enables parts of the GUI for when the user is entering
    * a new customer.
    */
   private void setGUIForCustomerEntry()
   {  newBtn.setEnabled(false);
      insertBtn.setEnabled(true);
      cancelBtn.setEnabled(true);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      ordersBtn.setEnabled(false);
      custList.setEnabled(false);
      currentCustomer = null;
      displayCurrentCustomer();
      nameTF.setEnabled(true);
      houseStreetTF.setEnabled(true);
      townCountyTF.setEnabled(true);
      teleNumTF.setEnabled(true);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The customer collection that this GUI presents to the users.
    */
   private CustomerCollection custCollection;

   /**
    * The CD collection that this GUI presents to the users.
    * Supplied in this class' constructor, and used when launching
    * an order GUI.)
    */
   private CDCollection cdCollection;

   /**
    * The customer whose details are currently being displayed, if any.
    */
   private Customer currentCustomer;

   /**
    * A subset of the customer collection that is currently being listed
    * on the screen.
    */
   private Vector viewedCustomers;

   /**
    * A JLIst that displays a list of some or all of the customers in the
    * customer collection, specifically those in viewedCustomers.
    */
   private JList custList;

   /**
    * A textfield into which a user can type a customer identification number;
    * a search for a customer is done using this number.
    */
   private JTextField searchIdNumTF;

   /**
    * A textfield into which a user can type a customer name; a search
    * for a customer is done using this name.
    */
   private JTextField searchNameTF;

   /**
    * A button for launching a search for a customer.
    */
   private JButton searchBtn;

   /**
    * A button for restoring the JList so that, instead of displaying
    * some subset of the customers in the collection (the results of some
    * search), it displays the full collection.
    */
   private JButton fullBtn;

   /**
    * The listener for fullBtn.
    */
   private ActionListener fullBtnLstnr;

   /**
    * A textfield for displaying the identification number of the current 
    * customer.
    */
   private JTextField idNumTF;

   /**
    * A textfield for displaying the name of the current customer.
    */
   private JTextField nameTF;

   /**
    * A textfield for displaying the address (house number and street
    * name) of the current customer.
    */
   private JTextField houseStreetTF;

   /**
    * A textfield for displaying the address (town and country) of the 
    * current customer.
    */
   private JTextField townCountyTF;

   /**
    * A textfield for displaying the telephone number of the current customer.
    */
   private JTextField teleNumTF;

   /**
    * A button for requesting that the screen be set up for insertion of
    * details of a new customer.
    */
   private JButton newBtn;

   /**
    * A button for indicating that entry of the details of a new customer is
    * complete, the details should be taken from the screen and a new
    * customer created and added to the collection.
    */
   private JButton insertBtn;

   /**
    * A button for aborting entry of a new customer.
    */
   private JButton cancelBtn;

   /**
    * A button for indicating that changes made to the details of the
    * current customer should be taken from the screen and used to update the
    * object in memory.
    */
   private JButton modifyBtn;

   /**
    * A button for requesting deletion of the current customer.
    */
   private JButton deleteBtn;

   /**
    * A button for requesting launch of an OrderGUI, so that the user
    * can view this customer's orders.
    */
   private JButton ordersBtn;

   /**
    * A button for requesting termination of this dialog, and return to
    * the parent JFrame or JDialog.
    */
   private JButton returnBtn;

   /**
    * The width of the identification number textfields.
    */
   private static final int ID_TF_COLS = 6;

   /**
    * The width of the name textfields.
    */
   private static final int NAME_TF_COLS = 30;

   /**
    * The width of the address textfields.
    */
   private static final int ADDRESS_TF_COLS = 30;

   /**
    * The width of the telephone number textfield.
    */
   private static final int TELE_TF_COLS = 10;

   /**
    * The text on searchBtn.
    */
   private static final String SEARCH = "SEARCH";

   /**
    * The text on fullBtn.
    */
   private static final String FULL = "See Full List";

   /**
    * The text on newBtn.
    */
   private static final String NEW = "NEW CUSTOMER";

   /**
    * The text on insertBtn.
    */
   private static final String INSERT = "INSERT";

   /** 
    * The text on cancelBtn.
    */
   private static final String CANCEL = "CANCEL";

   /**
    * The text on saveBtn.
    */
   private static final String MODIFY = "Save Changes";

   /**
    * The text on deleteBtn.
    */
   private static final String DELETE = "DELETE";

   /**
    * The text on orderBtn.
    */
   private static final String ORDERS = "ORDERS";

   /**
    * The text on returnBtn.
    */
   private static final String RETURN = "RETURN";
}
