import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.*;

/**
 * A class that represents GUIs for inserting, modifying, deleting
 * and selecting CDs from a CD collection.
 * @author Derek Bridge
 */
public class CDGUI
   extends JDialog
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new CD GUI object for a specified collection of
    * CDs.
    *
    * @param theParent the component that launched this GUI (a JFrame)
    * @param theCDCollection the collection of CDs.
    */
   public CDGUI(Frame theParent, CDCollection theCDCollection)
   {  super(theParent, "CD-Direkt CDs", true);
      initialise(theParent, theCDCollection);
   }

   /**
    * Allocates a new CD GUI object for a specified collection of
    * CDs.
    *
    * @param theParent the component that launched this GUI (a JDialog)
    * @param theCDCollection the collection of CDs.
    */
   public CDGUI(Dialog theParent, CDCollection theCDCollection)
   {  super(theParent, "CD-Direkt CDs", true);
      initialise(theParent, theCDCollection);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Setters----------------------------------------------------------- */

   /**
    * Displays the current CD's details (if any) on the CD panel.
    */
   public void displayCurrentCD()
   {  if (currentCD == null)
      {  idNumTF.setText("");
         artistTF.setText("");
         titleTF.setText("");
         priceTF.setText("");
         tracksList.setListData(new Vector());
      }
      else
      {  idNumTF.setText(currentCD.getIdNum());
         artistTF.setText(currentCD.getArtist());
         titleTF.setText(currentCD.getTitle());
         priceTF.setText("" + currentCD.getPrice());
         tracksList.setListData(currentCD.getTracks());
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Initialises a new CD GUI object for a specified collection of
    * CDs. (This method is a helper for two constructors.)
    *
    * @param theParent the component that launched this GUI (a JDialog
    * or JFrame).
    * @param theCDCollection the collection of CDs.
    */
   private void initialise(Component theParent, 
      CDCollection theCDCollection)
   {  parentComp = theParent;
      cdCollection = theCDCollection;
      JPanel searchPanel = createSearchPanel();
      JPanel cdPanel = createCDPanel();
      JPanel returnPanel = createReturnPanel();
      Container contentPane = getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(searchPanel, BorderLayout.WEST);
      contentPane.add(cdPanel, BorderLayout.EAST);
      contentPane.add(returnPanel, BorderLayout.SOUTH);
      setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
      pack();
   }

   /**
    * Lays out a panel containing a list of current CDs, and
    * offers search options.
    */
   private JPanel createSearchPanel()
   {  viewedCDs = cdCollection.findAll();
      cdList = new JList(viewedCDs);
      cdList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(cdList);
      JLabel idNumLbl = new JLabel("CD number:");
      searchIdNumTF = new JTextField("", ID_TF_COLS);
      JLabel artistLbl = new JLabel("Artist:");
      searchArtistTF = new JTextField("", ARTIST_TF_COLS);
      JLabel titleLbl = new JLabel("Title:");
      searchTitleTF = new JTextField("", TITLE_TF_COLS);
      searchBtn = new JButton(SEARCH);
      fullBtn = new JButton(FULL);

      cdList.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting()) // mouse wasn't yet at rest
            {  // do nothing
            }
            if (cdList.isSelectionEmpty()) // somehow, no selection was made
            {  // do nothing
            }
            else
            {  int index = cdList.getSelectedIndex(); 
               currentCD = (CD) viewedCDs.get(index);
               setGUIForCDViewing();
            }
          }
      });

      searchBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String idNum = searchIdNumTF.getText().trim();
            String artist = searchArtistTF.getText().trim();
            String title = searchTitleTF.getText().trim();
            if (! idNum.equals("") && artist.equals("") && title.equals(""))
            {  viewedCDs = cdCollection.findById(idNum);
               cdList.setListData(viewedCDs);
               cdList.setSelectedIndex(-1);
            }
            else if (idNum.equals("") && ! artist.equals("") && 
               title.equals(""))
            {  viewedCDs = cdCollection.findByArtist(artist);
               cdList.setListData(viewedCDs);
               cdList.setSelectedIndex(-1);
            }
            else if (idNum.equals("") && artist.equals("") && 
               ! title.equals(""))
            {  viewedCDs = cdCollection.findByTitle(title);
               cdList.setListData(viewedCDs);
               cdList.setSelectedIndex(-1);
            }
            else
            {  JOptionPane.showMessageDialog(null,
                  "To search, you must supply either a CD " +
                  "identification number or an artist or a title.",
                  "Error in search values",
                  JOptionPane.ERROR_MESSAGE);
            }
         }
      });

      fullBtnLstnr = new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  searchIdNumTF.setText("");
            searchArtistTF.setText("");
            searchTitleTF.setText("");
            viewedCDs = cdCollection.findAll();
            cdList.setListData(viewedCDs);
            cdList.setSelectedIndex(-1);
         }
      };
      fullBtn.addActionListener(fullBtnLstnr);

      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      GuiUtility.addComp(scrollPane, p, gbl, gbc, 0, 0, 
         1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(searchIdNumTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
      GuiUtility.addComp(artistLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(searchArtistTF, p, gbl, gbc, 2, 2, 1, 1, 0, 0);
      GuiUtility.addComp(titleLbl, p, gbl, gbc, 3, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 1, 1, 1, 0, 0);
      GuiUtility.addComp(searchTitleTF, p, gbl, gbc, 3, 2, 1, 1, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      GuiUtility.addComp(searchBtn, p, gbl, gbc, 4, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(fullBtn, p, gbl, gbc, 5, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Lays out a panel containing a selected CD's details, and
    * offers update options.
    */
   private JPanel createCDPanel()
   {  JLabel idNumLbl = new JLabel("CD number:");
      idNumTF = new JTextField("", ID_TF_COLS);
      idNumTF.setEnabled(false);
      JLabel artistLbl = new JLabel("Artist:");
      artistTF = new JTextField("", ARTIST_TF_COLS);
      JLabel titleLbl = new JLabel("Title:");
      titleTF = new JTextField("", TITLE_TF_COLS);
      JLabel priceLbl = new JLabel("Price:");
      priceTF = new JTextField("", PRICE_TF_COLS);
      newBtn = new JButton(NEW);
      insertBtn = new JButton(INSERT);
      cancelBtn = new JButton(CANCEL);
      modifyBtn = new JButton(MODIFY);
      deleteBtn = new JButton(DELETE);
      JLabel tracksLbl = new JLabel("Tracks:");
      tracksList = new JList(new Vector());
      tracksList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(tracksList);
      deleteTrackBtn = new JButton(DELETE_TRACK);
      newTrackTF = new JTextField("", TRACK_TF_COLS);
      saveTrackBtn = new JButton(SAVE_TRACK);
      setGUIForNoCDSelected();

      newBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForCDEntry();
         }
      });

      insertBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String artist = artistTF.getText().trim();
            String title = titleTF.getText().trim();
            int price = 0;
            try
            {  price = Integer.parseInt(priceTF.getText().trim());
            }
            catch (NumberFormatException nfe)
            {  price = 0;
            }
            if (artist.equals("") || title.equals("") ||
                price <= 0)
            {  JOptionPane.showMessageDialog(null,
                  "You must supply an artist and title, and a price " +
                  "that is greater than zero.", "Incomplete new CD data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentCD = 
                  new CD(artist, title, price);
               cdCollection.add(currentCD);
               setGUIForCDViewing();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      });

      cancelBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForNoCDSelected();
         }
      });

      modifyBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String artist = artistTF.getText().trim();
            String title = titleTF.getText().trim();
            int price = 0;
            try
            {  price = Integer.parseInt(priceTF.getText().trim());
            }
            catch (NumberFormatException nfe)
            {  price = 0;
            }
            if (artist.equals("") || title.equals("") ||
                price <= 0)
            {  JOptionPane.showMessageDialog(null,
                  "The artist and title cannot be blank and the price must " +
                  "be greater than zero.", "Incorrect changes to data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentCD.setArtist(artist);
               currentCD.setTitle(title);
               currentCD.setPrice(price);
               setGUIForCDViewing();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      });     

      deleteBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this CD. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
            if (confirm == JOptionPane.YES_OPTION)
            {  cdCollection.remove(currentCD);
               setGUIForNoCDSelected();
               fullBtnLstnr.actionPerformed(new ActionEvent(
                  fullBtn, ActionEvent.ACTION_PERFORMED, FULL));
            }
         }
      }); 

      deleteTrackBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = tracksList.getSelectedIndex();
            if (index == -1)
            {  JOptionPane.showMessageDialog(null,
                  "Select a track from the list above before clicking " +
                  "on the Delete Track button.",
                  "No track selected",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this track. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
               if (confirm == JOptionPane.YES_OPTION)
               {  currentCD.getTracks().removeElementAt(index);
                  setGUIForCDViewing();
               }
            }
         }
      });

      saveTrackBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String newTrack = newTrackTF.getText().trim();
            if (newTrack.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "Type a non-blank track name before clicking " +
                  "on the Save Track button.",
                  "No track entered",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentCD.addTrack(newTrack);
               setGUIForCDViewing();
            }
         }
      });

      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(idNumTF, p, gbl, gbc, 
         0, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(artistLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(artistTF, p, gbl, gbc, 
         1, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(titleLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(titleTF, p, gbl, gbc, 
         2, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(priceLbl, p, gbl, gbc, 3, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 1, 1, 1, 0, 0);
      GuiUtility.addComp(priceTF, p, gbl, gbc, 
         3, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl1 = new JPanel();
      btnPnl1.setLayout(new BoxLayout(btnPnl1, BoxLayout.X_AXIS));
      btnPnl1.add(newBtn);
      btnPnl1.add(insertBtn);
      btnPnl1.add(cancelBtn);
      JPanel btnPnl2 = new JPanel();
      btnPnl2.setLayout(new BoxLayout(btnPnl2, BoxLayout.X_AXIS));
      btnPnl2.add(modifyBtn);
      btnPnl2.add(deleteBtn);
      GuiUtility.addComp(btnPnl1, p, gbl, gbc, 4, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl2, p, gbl, gbc, 5, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.NORTHWEST;
      GuiUtility.addComp(tracksLbl, p, gbl, gbc, 6, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         6, 1, 1, 1, 0, 0);
      GuiUtility.addComp(scrollPane, p, gbl, gbc, 
         6, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl3 = new JPanel();
      btnPnl3.setLayout(new BoxLayout(btnPnl3, BoxLayout.X_AXIS));
      btnPnl3.add(deleteTrackBtn);
      GuiUtility.addComp(btnPnl3, p, gbl, gbc, 7, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      JPanel btnPnl4 = new JPanel();
      GridBagLayout gblP = new GridBagLayout();
      GridBagConstraints gbcP = new GridBagConstraints();
      gbcP.fill = GridBagConstraints.NONE;
      gbcP.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(newTrackTF, btnPnl4, gblP, gbcP,
         0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(saveTrackBtn, btnPnl4, gblP, gbcP, 
         0, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl4, p, gbl, gbc, 8, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Creates the button that takes the user back to the parent
    * component, i.e. the frame from which s/he came before viewing
    * this frame.
    */
   private JPanel createReturnPanel()
   {  returnBtn = new JButton(RETURN);
      returnBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setVisible(false);
            if (parentComp instanceof OrderGUI)
            {  OrderGUI orderGUI = (OrderGUI) parentComp;
               if (currentCD != null)
               {  orderGUI.setChosenCD(currentCD);
               }
            }
            dispose();
         }
      });
      JPanel p = new JPanel();
      p.add(returnBtn);
      return p;
   }

   /**
    * Disables/enables parts of the GUI for when the user has no
    * CD selected.
    */
   private void setGUIForNoCDSelected()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      cdList.setEnabled(true);
      currentCD = null;
      displayCurrentCD();
      artistTF.setEnabled(false);
      titleTF.setEnabled(false);
      priceTF.setEnabled(false);
      tracksList.setEnabled(false);
      deleteTrackBtn.setEnabled(false);
      newTrackTF.setText("");
      newTrackTF.setEnabled(false);
      saveTrackBtn.setEnabled(false);
   }

   /**
    * Disables/enables parts of the GUI for when the user is viewing
    * a CD.
    */
   private void setGUIForCDViewing()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(true);
      deleteBtn.setEnabled(true);
      cdList.setEnabled(true);
      displayCurrentCD();
      artistTF.setEnabled(true);
      titleTF.setEnabled(true);
      priceTF.setEnabled(true);
      tracksList.setEnabled(true);
      deleteTrackBtn.setEnabled(true);
      newTrackTF.setText("");
      newTrackTF.setEnabled(true);
      saveTrackBtn.setEnabled(true);
   }

   /**
    * Disables/enables parts of the GUI for when the user is entering
    * a new CD.
    */
   private void setGUIForCDEntry()
   {  newBtn.setEnabled(false);
      insertBtn.setEnabled(true);
      cancelBtn.setEnabled(true);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      cdList.setEnabled(false);
      currentCD = null;
      displayCurrentCD();
      artistTF.setEnabled(true);
      titleTF.setEnabled(true);
      priceTF.setEnabled(true);
      tracksList.setEnabled(false);
      deleteTrackBtn.setEnabled(false);
      newTrackTF.setText("");
      newTrackTF.setEnabled(false);
      saveTrackBtn.setEnabled(false);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The JFrame or JDialog from which this GUI was launched.
    */
   private Component parentComp;

   /**
    * The CD collection that this GUI presents to the users.
    */
   private CDCollection cdCollection;

   /**
    * The CD whose details are currently being displayed, if any.
    */
   private CD currentCD;

   /**
    * A subset of the CD collection that is currently being listed
    * on the screen.
    */
   private Vector viewedCDs;

   /**
    * A JLIst that displays a list of some or all of the CDs in the
    * CD collection, specifically those in viewedCDs.
    */
   private JList cdList;

   /**
    * A textfield into which a user can type a CD identification number;
    * a search for a CD is done using this number.
    */
   private JTextField searchIdNumTF;

   /**
    * A textfield into which a user can type a CD artist name; a search
    * for a CD is done using this name.
    */
   private JTextField searchArtistTF;

   /**
    * A textfield into which a user can type a CD title; a search for a
    * CD is done using this title.
    */
   private JTextField searchTitleTF;

   /**
    * A button for launching a search for a CD.
    */
   private JButton searchBtn;

   /**
    * A button for restoring the JList so that, instead of displaying
    * some subset of the CDs in the collection (the results of some
    * search), it displays the full collection.
    */
   private JButton fullBtn;
 
   /**
    * The listener for fullBtn.
    */
   private ActionListener fullBtnLstnr;

   /**
    * A textfield for displaying the identification number of the current CD.
    */
   private JTextField idNumTF;

   /**
    * A textfield for displaying the artist name for the current CD.
    */
   private JTextField artistTF;

   /**
    * A textfield for displaying the title of the current CD.
    */
   private JTextField titleTF;

   /**
    * A textfield for displaying the price of the current CD.
    */
   private JTextField priceTF;

   /**
    * A button for requesting that the screen be set up for insertion of
    * details of a new CD.
    */
   private JButton newBtn;

   /**
    * A button for indicating that entry of the details of a new CD is
    * complete, the details should be taken from the screen and a new
    * CD created and added to the collection.
    */
   private JButton insertBtn;
   
   /**
    * A button for aborting entry of a new CD.
    */
   private JButton cancelBtn;

   /**
    * A button for indicating that changes made to the details of the
    * current CD should be taken from the screen and used to update the
    * object in memory.
    */
   private JButton modifyBtn;

   /**
    * A button for requesting deletion of the current CD.
    */
   private JButton deleteBtn;

   /**
    * A JList for display of the tracks of the current CD.
    */
   private JList tracksList;

   /**
    * A button for requesting that the track selected in the tracksList
    * JLIst be deleted.
    */
   private JButton deleteTrackBtn;

   /**
    * A textfield for entry of a new track for the current CD.
    */
   private JTextField newTrackTF;

   /**
    * A button for indicating that entry of a new track is complete,
    * details should be taken from the screen and used to update the
    * current CD in memory.
    */
   private JButton saveTrackBtn;

   /**
    * A button for requesting termination of this dialog, and return to
    * the parent JFrame or JDialog.
    */
   private JButton returnBtn;

   /**
    * The width of the identification number textfields.
    */
   private static final int ID_TF_COLS = 6;

   /**
    * The width of the artist name textfields.
    */
   private static final int ARTIST_TF_COLS = 30;

   /**
    * The width of the CD title textfields.
    */
   private static final int TITLE_TF_COLS = 30;

   /**
    * The width of the price textfield.
    */
   private static final int PRICE_TF_COLS = 6;

   /**
    * The width of the track title textfield.
    */
   private static final int TRACK_TF_COLS = 30;

   /**
    * The text on searchBtn.
    */
   private static final String SEARCH = "SEARCH";

   /**
    * The text on fullBtn.
    */
   private static final String FULL = "See Full List";

   /**
    * The text on newBtn.
    */
   private static final String NEW = "NEW CD";

   /**
    * The text on insertBtn.
    */
   private static final String INSERT = "INSERT";

   /** 
    * The text on cancelBtn.
    */
   private static final String CANCEL = "CANCEL";

   /**
    * The text on saveBtn.
    */
   private static final String MODIFY = "Save Changes";

   /**
    * The text on deleteBtn.
    */
   private static final String DELETE = "DELETE";

   /**
    * The text on deleteTrackBtn.
    */
   private static final String DELETE_TRACK = "Delete Track";

   /**
    * The text on saveTrackbtn.
    */
   private static final String SAVE_TRACK = "Save New Track";

   /**
    * The text on returnBtn.
    */
   private static final String RETURN = "RETURN";
}
