/**
 * A class that represents fractions (rational numbers)
 * @author Derek Bridge
 */
public class Fraction
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new Fraction object with the given numerator and
    * denominator.
    * To allocate a negative fraction, either parameter may be negative.
    * But if both are negative, a positive fraction is allocated.
    *
    * @param aNum the numerator of this fraction.
    * @param aDenom the denominator of this fraction; cannot be zero.
    */
   public Fraction(int aNum, int aDenom)
   {  /* If fraction represents zero, the
         denominator is always reduced to 1
       */
      if (aNum == 0)
      {  num = 0;
         denom = 1;
      }
      else 
      {  int n = aNum; // the new numerator.
         int d = aDenom; // the new denominator.
         /* Reduce n and d using their gcd.
            The gcd method requires +ve args,
            so pass in absolute values.
          */
         int g = gcd(Math.abs(aNum), Math.abs(aDenom));
         n = n / g;
         d = d / g;
         /* If denom is negative, move its 
            sign to the numerator.
          */
         if (aDenom < 0)
         {  n = - n;
            d = - d;
         }
         /* Now move n and d to the instance 
            variables.
          */
         num = n;
         denom = d;
      }
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns a new fraction that represents 
    * the sum of this fraction and the 
    * specified fraction.
    * @param aFraction the fraction we are 
    * adding; must not be null.
    * @return a fraction that represents the 
    * sum of this fraction and the parameter.
    */
   public Fraction plus(Fraction aFraction)
   {  return new Fraction( 
         this.num * aFraction.denom + aFraction.num * this.denom,
         denom * aFraction.denom);
   }

   /**
    * Returns a new fraction that represents 
    * the difference between this fraction and the 
    * specified fraction.
    * @param aFraction the fraction we are 
    * subtracting.
    * @return a fraction that represents the 
    * difference between this fraction and the parameter.
    */
   public Fraction minus(Fraction aFraction)
   {  return new Fraction(
         this.num * aFraction.denom - aFraction.num * this.denom,
         denom * aFraction.denom);
   }
                      
   /**
    * Determines whether the specified 
    * fraction is equal to this fraction. 
    * The result is true if and only if the 
    * specified fraction is non-null and 
    * denotes the same fraction as this 
    * fraction.
    *
    * @param aFraction the fraction to 
    * compare this fraction against.
    * @return true if the fractions are 
    * mathematically equal; false otherwise.
    */
   public boolean isEqualTo(Fraction aFraction)
   {  if (aFraction == null)
      {  return false;
      }
      else
      {  return (this.num == aFraction.num && 
            this.denom == aFraction.denom);
      }
   }

   /**
    * Returns a string representation of this 
    * fraction in reduced form, e.g. it 
    * returns "2/3", never "4/6".
    *
    * @return a string representation.
    */
   public String toString()
   {  return (num + "/" + denom); 
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Returns the greatest common divisor 
    * of two positive integers.
    *
    * @param a - any int; must be > 0.
    * @param b - any int; must be > 0.
    * @return the gcd of a and b.
    */
   private static int gcd(int a, int b)
   {  // Euclid's algorithm (recursive)
      if (b != 0)
      {  return gcd(b, a % b);
      }
      else
      {  return a;
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

/* =====================================
   INSTANCE VARIABLES & CLASS VARIABLES
   =====================================
*/

   /**
    * The fraction's numerator.
    * Fraction's are stored in reduced
    * form, e.g. 4/6 is stored as 2/3;
    * 0/7 is stored as 0/1.
    * The numerator will always carry the 
    * sign.
    */
   private int num;

   /**
    * The fraction's denominator.
    * Since the numerator carries the
    * sign, the denominator will always
    * be positive.
    */
   private int denom;

/* =======================================================================
       TEST DRIVER
   =======================================================================
*/
   public static void main(String[] args)
   {  Fraction f;
      String s;

      // Testing the constructor and toString

      f = new Fraction(2, 3);
      s = f.toString();
      if (! (s.equals("2/3")))
      {  System.out.println("Error: " + s + " should be 2/3");
      }

      /* Other test cases for the constructor might include, e.g.
         3/4, -2/3, -3/4, 5/4, -5/4, 4/6, -4/6, 6/4, -6/4,
         3/1, -3/1, 0/1, 3/3, -3/3, 1/1, -1/1, 0/3, 2/-3, -2/-3,
         1/0, 3/0, -3/0, 0/0
       */

      // Testing isEqualTo

      Fraction f1;
      Fraction f2;
      f1 = new Fraction(2, 3);
      f2 = new Fraction(2, 3);
      if (! f1.isEqualTo(f2))
      {  System.out.println("Error: " + f1.toString() + " and " + 
            f2.toString() + " not recognised as equal");
      }

      /* Other test cases will include asking whether 2/3 equals
         4/6 (true), whether 2/3 equals 4/5 (false, whether 2/3
         equals null, and so on.
       */

      // Testing plus by manual calculations (minus would be similar)

      Fraction f3;
      f1 = new Fraction(1, 2);
      f2 = new Fraction(1, 3);
      f3 = new Fraction(5, 6);
      if (! (f1.plus(f2).isEqualTo(f3)))
      {  System.out.println("Error: " + f1.toString() + " plus " + 
            f2.toString() +" not recognised as " + f3.toString());
      }

      // Testing plus/minus/isEqualTo using a maths property

      f1 = new Fraction(2, 3);
      f2 = new Fraction(1, 5);
      if (! (f1.plus(f2).minus(f1).isEqualTo(f2)))
      {  System.out.println("Error: " + f1.toString() + " plus " + 
            f2.toString() + " minus " + f1.toString() + 
            " not recognised as " + f2.toString());
      }

      // Random testing of plus/minus/isEqualTo

      for (int i = 0; i < 1000; i++)
      {  int r1 = (int) (Math.random() * 101 - 50);
         int r2 = (int) (Math.random() * 101 - 50);
         int r3 = (int) (Math.random() * 101 - 50);
         int r4 = (int) (Math.random() * 101 - 50);
         if (r2 != 0 && r4 != 0)
         {  f1 = new Fraction(r1, r2);
            f2 = new Fraction(r3, r4);
            if (! (f1.plus(f2).minus(f1).isEqualTo(f2)))
            {  System.out.println("Error: " + f1.toString() + 
                  " plus " + f2.toString() + " minus " + f1.toString() + 
                  " not recognised as " + f2.toString());
            }
         }
      }
   }
}
