/**
 * A class that represents simple time objects.
 *
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class Time
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new time object for the specified hour,
    * minutes and seconds.
    *
    * @param theHour this time's hour (24-hour clock)
    * @param theMins this time's minutes past the hour
    * @param theSecs this time's seconds past the minute
    */
   public Time(int theHour, int theMins, int theSecs)
   {  timeInSecs = theHour * SECS_PER_HOUR +
         theMins * MINS_PER_HOUR + theSecs;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Public class constants-------------------------------------------- */

   /**
    * The number of hours per day.
    */
   public static final int HOURS_PER_DAY = 24;

   /**
    * The number of seconds per minute.
    */
   public static final int SECS_PER_MIN = 60;

   /**
    * The number of minutes per hour.
    */
   public static final int MINS_PER_HOUR = 60;

   /**
    * The number of seconds per hour.
    */
   public static final int SECS_PER_HOUR = SECS_PER_MIN * MINS_PER_HOUR;

/* --Getters----------------------------------------------------------- */

   /**
    * Returns true if this time is equal to the specified time; 
    * false otherwise.
    *
    * @param aTime the time being compared with this time.
    * @return true if and only if this time equals the specified 
    * time.
    */
   public boolean isEqualsTo(Time aTime)
   {  return this.timeInSecs == aTime.timeInSecs;
   }

   /**
    * Returns true if this time is before the specified time;
    * false otherwise.
    *
    * @param aTime the time being compared with this time.
    * @return true if and only if this time is strictly earlier
    * than the specified time.
    */
   public boolean isBefore(Time aTime)
   {  return this.timeInSecs < aTime.timeInSecs;
   }

   /**
    * Returns the number of seconds there are between this time
    * and the specified time. 
    *
    * @param aTime the end time of the interval that is being
    * computed. (It is assumed that aTime is after this time.)
    * @return the number of seconds there are between this time
    * and the specified time.
    */
   public int getIntervalInSecsUntil(Time aTime)
   {  return aTime.timeInSecs - this.timeInSecs;
   }

   /**
    * Returns a string that expresses this time as a `clock time',
    * i.e. hours, minutes and seconds.
    *
    * @return this time as a `clock time'.
    */
   public String getClockTime()
   {  return timeInSecs / SECS_PER_HOUR + ":" +
         timeInSecs % SECS_PER_HOUR / SECS_PER_MIN +
         ":" + timeInSecs % SECS_PER_HOUR % SECS_PER_MIN;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   /**
    * This time expressed as seconds since midnight.
    */
   private int timeInSecs; 
}



   