import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.util.*;

/**
 * A class that represents GUIs for inserting, modifying, deleting
 * and selecting customer orders.
 * @author Derek Bridge
 */
public class OrderGUI
   extends JFrame
   implements ICDGUI
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new Order GUI object for a specified collection
    * of orders.
    *
    * @param theParent the component that launched this GUI
    * @param theOrderCollection the collection of orders.
    * @param theCDCollection the collection of CDs.
    */
   public OrderGUI(ICDGUI theParent, 
      OrderCollection theOrderCollection, CDCollection theCDCollection)
   {  this(theParent, null, theOrderCollection, theCDCollection);
   }

   /**
    * Allocates a new Order GUI object for a specified customer's orders.
    *
    * @param theParent the component that launched this GUI
    * @param theCust the customer whose orders we are to view.
    * @param theOrderCollection the collection of orders.
    * @param theCDCollection the collection of CDs.
    */
   public OrderGUI(ICDGUI theParent, Customer theCust,
      OrderCollection theOrderCollection, CDCollection theCDCollection)
   {  super("CD-Direkt Orders");
      parentComp = theParent;
      currentCustomer = theCust;
      orderCollection = theOrderCollection;
      cdCollection = theCDCollection;
      JPanel searchPanel = createSearchPanel();
      JPanel orderPanel = createOrderPanel();
      JPanel returnPanel = createReturnPanel();
      Container contentPane = getContentPane();
      contentPane.setLayout(new BorderLayout());
      contentPane.add(searchPanel, BorderLayout.WEST);
      contentPane.add(orderPanel, BorderLayout.EAST);
      contentPane.add(returnPanel, BorderLayout.SOUTH);
      setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
      pack();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Setters----------------------------------------------------------- */

   /**
    * Displays the current order's details (if any) on the Order panel.
    */
   public void displayCurrentOrder()
   {  if (currentCustomer == null)
      {  custIdNumTF.setText("");
         custNameTF.setText("");
         custHouseStreetTF.setText("");
         custTownCountyTF.setText("");
         custTeleNumTF.setText("");
      }
      else
      {  custIdNumTF.setText(currentCustomer.getIdNum());
         custNameTF.setText(currentCustomer.getName());
         custHouseStreetTF.setText(currentCustomer.getHouseStreet());
         custTownCountyTF.setText(currentCustomer.getTownCounty());
         custTeleNumTF.setText(currentCustomer.getTeleNum());
      }
      if (currentOrder == null)
      {  idNumTF.setText("");
         dateTF.setText("");
         salesPersonTF.setText("");
         statusTF.setText("");
         recipientTF.setText("");
         shipHouseStreetTF.setText("");
         shipTownCountyTF.setText("");
         ccTypeTF.setText("");
         ccNumTF.setText("");
         ccNameTF.setText("");
         ccExpiryTF.setText("");
         linesList.setListData(new Vector());
      }
      else
      {  idNumTF.setText(currentOrder.getIdNum());
         dateTF.setText(currentOrder.getDate());
         salesPersonTF.setText(currentOrder.getSalesPerson());
         statusTF.setText(currentOrder.getStatus());
         recipientTF.setText(currentOrder.getRecipient());
         shipHouseStreetTF.setText(currentOrder.getShipHouseStreet());
         shipTownCountyTF.setText(currentOrder.getShipTownCounty());
         ccTypeTF.setText(currentOrder.getCCType());
         ccNumTF.setText(currentOrder.getCCNum());
         ccNameTF.setText(currentOrder.getCCName());
         ccExpiryTF.setText(currentOrder.getCCExpiry());
         linesList.setListData(currentOrder.getOrderLines());
      }
   }

   /**
    * Disables this GUI when the user goes to another GUI to choose
    * a CD to add to the order.
    */
   public void disableGUI()
   {  newBtn.setEnabled(false);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      orderList.setEnabled(false);
      salesPersonTF.setEnabled(false);
      dateTF.setEnabled(false);
      statusTF.setEnabled(false);
      recipientTF.setEnabled(false);
      shipHouseStreetTF.setEnabled(false);
      shipTownCountyTF.setEnabled(false);
      ccTypeTF.setEnabled(false);
      ccNumTF.setEnabled(false);
      ccNameTF.setEnabled(false);
      ccExpiryTF.setEnabled(false);
      linesList.setEnabled(false);
      deleteLineBtn.setEnabled(false);
      cdSearchBtn.setEnabled(false);
      newQtyTF.setEnabled(false);
      saveLineBtn.setEnabled(false);
      returnBtn.setEnabled(false);
   }

   /**
    * Enable this GUI when the user returns from another GUI where
    * s/he was looking at the CDs s/he could add to this order.
    */
   public void enableGUI()
   {  setGUIForOrderViewing();
      returnBtn.setEnabled(true);
   }

   /**
    * If one GUI sends you to another GUI and that GUI wants to
    * supply a result, it does so using this setter. In order GUIs,
    * this will be CD the user has chosen on the CD GUI.
    *
    * @param anObj the result being supplied.
    */
   public void setResult(Object anObj)
   {  if (anObj != null)
      {  newCD = (CD) anObj;
         newCDIdNumTF.setText(newCD.getIdNum());
      }
   }
      

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Lays out a panel containing a list of current Orders.
    */
   private JPanel createSearchPanel()
   {  viewedOrders = orderCollection.findAll();
      orderList = new JList(viewedOrders);
      orderList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(orderList);

      orderList.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting()) // mouse wasn't yet at rest
            {  // do nothing
            }
            if (orderList.isSelectionEmpty()) // somehow, no selection was made
            {  // do nothing
            }
            else
            {  int index = orderList.getSelectedIndex(); 
               currentOrder = (Order) viewedOrders.elementAt(index);
               currentCustomer = currentOrder.getCustomer();
               setGUIForOrderViewing();
            }
          }
      });

      JPanel p = new JPanel();
      p.add(scrollPane);
      return p;
   }

   /**
    * Lays out a panel containing a selected Order's details, and
    * offers update options.
    */
   private JPanel createOrderPanel()
   {  JLabel custIdNumLbl = new JLabel("Customer number:");
      custIdNumTF = new JTextField("", ID_TF_COLS);
      custIdNumTF.setEnabled(false);
      JLabel custNameLbl = new JLabel("Customer name:");
      custNameTF = new JTextField("", NAME_TF_COLS);
      custNameTF.setEnabled(false);
      JLabel custAddressLbl = new JLabel("Address:");
      custHouseStreetTF = new JTextField("", ADDRESS_TF_COLS);
      custHouseStreetTF.setEnabled(false);
      custTownCountyTF = new JTextField("", ADDRESS_TF_COLS);
      custTownCountyTF.setEnabled(false);
      JLabel custTeleNumLbl = new JLabel("Tel:");
      custTeleNumTF = new JTextField("", TELE_TF_COLS);
      custTeleNumTF.setEnabled(false);
      JLabel idNumLbl = new JLabel("Order number:");
      idNumTF = new JTextField("", ID_TF_COLS);
      idNumTF.setEnabled(false);
      JLabel dateLbl = new JLabel("Entry date:");
      dateTF = new JTextField("", DATE_TF_COLS);
      JLabel salesPersonLbl = new JLabel("Sales person:");
      salesPersonTF = new JTextField("", SALESPERSON_TF_COLS);
      JLabel statusLbl = new JLabel("Status:");
      statusTF = new JTextField("", STATUS_TF_COLS);
      JLabel recipientLbl = new JLabel("Ship to:");
      recipientTF = new JTextField("", NAME_TF_COLS);
      shipHouseStreetTF = new JTextField("", ADDRESS_TF_COLS);
      shipTownCountyTF = new JTextField("", ADDRESS_TF_COLS);
      JLabel ccTypeLbl = new JLabel("Credit card type:");
      ccTypeTF = new JTextField("", CCTYPE_TF_COLS);
      JLabel ccNumLbl = new JLabel("Credit card number:");
      ccNumTF = new JTextField("", CCNUM_TF_COLS);
      JLabel ccNameLbl = new JLabel("Credit card name:");
      ccNameTF = new JTextField("", NAME_TF_COLS);
      JLabel ccExpiryLbl = new JLabel("Credit card expiry:");
      ccExpiryTF = new JTextField("", DATE_TF_COLS);
      newBtn = new JButton(NEW);
      insertBtn = new JButton(INSERT);
      cancelBtn = new JButton(CANCEL);
      modifyBtn = new JButton(MODIFY);
      deleteBtn = new JButton(DELETE);
      JLabel linesLbl = new JLabel("Order lines:");
      linesList = new JList(new Vector());
      linesList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      JScrollPane scrollPane = new JScrollPane(linesList);
      deleteLineBtn = new JButton(DELETE_LINE);
      cdSearchBtn = new JButton(FIND_CD);
      newCDIdNumTF = new JTextField("", CDID_TF_COLS);
      newCDIdNumTF.setEnabled(false);
      newCD = null;
      JLabel newQtyLbl = new JLabel("Quantity:");
      newQtyTF = new JTextField("", QTY_TF_COLS);
      saveLineBtn = new JButton(SAVE_LINE);
      setGUIForNoOrderSelected();

      newBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForOrderEntry();
         }
      });

      insertBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String salesPerson = salesPersonTF.getText().trim();
            String date = dateTF.getText().trim();
            String status = statusTF.getText().trim();
            String recipient = recipientTF.getText().trim();
            String shipHouseStreet = shipHouseStreetTF.getText().trim();
            String shipTownCounty = shipTownCountyTF.getText().trim();
            String ccType = ccTypeTF.getText().trim();
            String ccNum = ccNumTF.getText().trim();
            String ccName = ccNameTF.getText().trim();
            String ccExpiry = ccExpiryTF.getText().trim();
            if (date.equals("") || status.equals("") ||
                ccType.equals("") || ccNum.equals("") ||
                ccName.equals("") || ccExpiry.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "You must supply a date, salesperson, all credit card " +
                  "details and, optionally, a complete shipping address.", 
                  "Incomplete new order data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  if (recipient.equals(""))
               {  recipient = currentCustomer.getName();
               }
               if (shipHouseStreet.equals("") || shipTownCounty.equals(""))
               {  shipHouseStreet = currentCustomer.getHouseStreet();
                  shipTownCounty = currentCustomer.getTownCounty();
               }
               currentOrder = new Order(currentCustomer, salesPerson, date, 
                  recipient, shipHouseStreet, shipTownCounty, ccType,
                  ccNum, ccName, ccExpiry);
               orderCollection.add(currentOrder);
               setGUIForOrderViewing();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      });

      cancelBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setGUIForNoOrderSelected();
         }
      });

      modifyBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String date = dateTF.getText().trim();
            String status = statusTF.getText().trim();
            String recipient = recipientTF.getText().trim();
            String shipHouseStreet = shipHouseStreetTF.getText().trim();
            String shipTownCounty = shipTownCountyTF.getText().trim();
            String ccType = ccTypeTF.getText().trim();
            String ccNum = ccNumTF.getText().trim();
            String ccName = ccNameTF.getText().trim();
            String ccExpiry = ccExpiryTF.getText().trim();
            if (date.equals("") || status.equals("") ||
                ccType.equals("") || ccNum.equals("") ||
                ccName.equals("") || ccExpiry.equals(""))
            {  JOptionPane.showMessageDialog(null,
                  "You must supply a date, salesperson and all credit card " +
                  "details.",
                  "Incorrect changes made to order data",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentOrder.setDate(date);
               currentOrder.setStatus(status);
               if (recipient.equals(""))
               {  recipient = currentCustomer.getName();
               }
               if (shipHouseStreet.equals("") || shipTownCounty.equals(""))
               {  shipHouseStreet = currentCustomer.getHouseStreet();
                  shipTownCounty = currentCustomer.getTownCounty();
               }
               currentOrder.setRecipient(recipient);
               currentOrder.setShipHouseStreet(shipHouseStreet);
               currentOrder.setShipTownCounty(shipTownCounty);
               currentOrder.setCCType(ccType);
               currentOrder.setCCNum(ccNum);
               currentOrder.setCCName(ccName);
               currentOrder.setCCExpiry(ccExpiry);
               setGUIForOrderViewing();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      });     

      deleteBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this Order. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
            if (confirm == JOptionPane.YES_OPTION)
            {  orderCollection.remove(currentOrder);
               currentCustomer.removeOrder(currentOrder);
               setGUIForNoOrderSelected();
               viewedOrders = orderCollection.findAll();
               orderList.setListData(viewedOrders);
               orderList.setSelectedIndex(-1);
            }
         }
      }); 

      deleteLineBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = linesList.getSelectedIndex();
            if (index == -1)
            {  JOptionPane.showMessageDialog(null,
                  "Select an order line from the list above before clicking " +
                  "on the Delete Line button.",
                  "No order line selected",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  int confirm = JOptionPane.showConfirmDialog(null,
               "This will permanently delete this order line. Are you sure?",
               "Confirm or cancel deletion",
               JOptionPane.YES_NO_OPTION);
               if (confirm == JOptionPane.YES_OPTION)
               {  currentOrder.getOrderLines().removeElementAt(index);
                  setGUIForOrderViewing();
               }
            }
         }
      });

      cdSearchBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  disableGUI();
            CDGUI cdGUI = new CDGUI(OrderGUI.this, cdCollection);
            cdGUI.setVisible(true);
         }
      });

      saveLineBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String newQtyStr = newQtyTF.getText().trim();
            int newQty = 0;
            try
            {  newQty = Integer.parseInt(newQtyStr);
            }
            catch (NumberFormatException nfe)
            {  newQty = 0;
            }
            if (newCD == null || newQty <= 0) 
            {  JOptionPane.showMessageDialog(null,
                  "Choose a CD and specify a quantity greater than zero " +
                  "before clicking on the Save New Order Line button.",
                  "Invlaid new order line",
                  JOptionPane.ERROR_MESSAGE);
            }
            else
            {  currentOrder.addOrderLine(new OrderLine(newCD, newQty));
               setGUIForOrderViewing();
            }
         }
      });

      JPanel p = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(custIdNumLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custIdNumTF, p, gbl, gbc, 
         0, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custNameLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         1, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custNameTF, p, gbl, gbc, 
         1, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custAddressLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      GuiUtility.addComp(custHouseStreetTF, p, gbl, gbc, 
         2, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(custTownCountyTF, p, gbl, gbc, 3, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 3, 1, 1, 0, 0);
      GuiUtility.addComp(custTeleNumLbl, p, gbl, gbc, 3, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         3, 5, 1, 1, 0, 0);
      GuiUtility.addComp(custTeleNumTF, p, gbl, gbc, 
         3, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(idNumLbl, p, gbl, gbc, 4, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 1, 1, 1, 0, 0);
      GuiUtility.addComp(idNumTF, p, gbl, gbc, 
         4, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 3, 1, 1, 0, 0);
      GuiUtility.addComp(salesPersonLbl, p, gbl, gbc, 4, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         4, 5, 1, 1, 0, 0);
      GuiUtility.addComp(salesPersonTF, p, gbl, gbc, 
         4, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(dateLbl, p, gbl, gbc, 5, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 1, 1, 1, 0, 0);
      GuiUtility.addComp(dateTF, p, gbl, gbc, 
         5, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 3, 1, 1, 0, 0);
      GuiUtility.addComp(statusLbl, p, gbl, gbc, 5, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         5, 5, 1, 1, 0, 0);
      GuiUtility.addComp(statusTF, p, gbl, gbc, 
         5, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(recipientLbl, p, gbl, gbc, 6, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         6, 1, 1, 1, 0, 0);
      GuiUtility.addComp(recipientTF, p, gbl, gbc, 
         6, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(shipHouseStreetTF, p, gbl, gbc, 
         7, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(shipTownCountyTF, p, gbl, gbc, 
         8, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(ccTypeLbl, p, gbl, gbc, 9, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 1, 1, 1, 0, 0);
      GuiUtility.addComp(ccTypeTF, p, gbl, gbc, 
         9, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 3, 1, 1, 0, 0);
      GuiUtility.addComp(ccNumLbl, p, gbl, gbc, 9, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         9, 5, 1, 1, 0, 0);
      GuiUtility.addComp(ccNumTF, p, gbl, gbc, 
         9, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(ccNameLbl, p, gbl, gbc, 10, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 1, 1, 1, 0, 0);
      GuiUtility.addComp(ccNameTF, p, gbl, gbc, 
         10, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 3, 1, 1, 0, 0);
      GuiUtility.addComp(ccExpiryLbl, p, gbl, gbc, 10, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         10, 5, 1, 1, 0, 0);
      GuiUtility.addComp(ccExpiryTF, p, gbl, gbc, 
         10, 6, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl1 = new JPanel();
      btnPnl1.setLayout(new BoxLayout(btnPnl1, BoxLayout.X_AXIS));
      btnPnl1.add(newBtn);
      btnPnl1.add(insertBtn);
      btnPnl1.add(cancelBtn);
      JPanel btnPnl2 = new JPanel();
      btnPnl2.setLayout(new BoxLayout(btnPnl2, BoxLayout.X_AXIS));
      btnPnl2.add(modifyBtn);
      btnPnl2.add(deleteBtn);
      GuiUtility.addComp(btnPnl1, p, gbl, gbc, 11, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl2, p, gbl, gbc, 12, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.NORTHWEST;
      GuiUtility.addComp(linesLbl, p, gbl, gbc, 13, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         13, 1, 1, 1, 0, 0);
      GuiUtility.addComp(scrollPane, p, gbl, gbc, 
         13, 2, 1, GridBagConstraints.REMAINDER, 0, 0);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      JPanel btnPnl3 = new JPanel();
      btnPnl3.setLayout(new BoxLayout(btnPnl3, BoxLayout.X_AXIS));
      btnPnl3.add(deleteLineBtn);
      GuiUtility.addComp(btnPnl3, p, gbl, gbc, 14, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      JPanel btnPnl4 = new JPanel();
      GridBagLayout gblP = new GridBagLayout();
      GridBagConstraints gbcP = new GridBagConstraints();
      gbcP.fill = GridBagConstraints.NONE;
      gbcP.anchor = GridBagConstraints.WEST;
      GuiUtility.addComp(cdSearchBtn, btnPnl4, gblP, gbcP, 0, 0, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 1, 1, 1, 0, 0);
      GuiUtility.addComp(newCDIdNumTF, btnPnl4, gblP, gbcP,
         0, 2, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 3, 1, 1, 0, 0);
      GuiUtility.addComp(newQtyLbl, btnPnl4, gblP, gbcP, 0, 4, 1, 1, 0, 0);
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 5, 1, 1, 0, 0);
      GuiUtility.addComp(newQtyTF, btnPnl4, gblP, gbcP, 
         0, 6, 1, GridBagConstraints.REMAINDER, 0, 0); 
      GuiUtility.addComp(Box.createHorizontalStrut(10), btnPnl4, gblP, gbcP, 
         0, 7, 1, 1, 0, 0);
      GuiUtility.addComp(saveLineBtn, btnPnl4, gblP, gbcP, 
         0, 8, 1, GridBagConstraints.REMAINDER, 0, 0);
      GuiUtility.addComp(btnPnl4, p, gbl, gbc, 15, 0,
         1, GridBagConstraints.REMAINDER, 0, 0);
      return p;
   }

   /**
    * Creates the button that takes the user back to the parent
    * component, i.e. the frame from which s/he came before viewing
    * this frame.
    */
   private JPanel createReturnPanel()
   {  returnBtn = new JButton(ICDGUI.RETURN);
      returnBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  setVisible(false);
            parentComp.enableGUI();
            dispose();
         }
      });
      JPanel p = new JPanel();
      p.add(returnBtn);
      return p;
   }

   /**
    * Disables/enables parts of the GUI for when the user has no
    * customer/order selected.
    */
   private void setGUIForNoOrderSelected()
   {  if (currentCustomer == null)
      {  newBtn.setEnabled(false);
      }
      else
      {  newBtn.setEnabled(true);
      }
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      orderList.setEnabled(true);
      currentOrder = null;
      displayCurrentOrder();
      salesPersonTF.setEnabled(false);
      dateTF.setEnabled(false);
      statusTF.setEnabled(false);
      recipientTF.setEnabled(false);
      shipHouseStreetTF.setEnabled(false);
      shipTownCountyTF.setEnabled(false);
      ccTypeTF.setEnabled(false);
      ccNumTF.setEnabled(false);
      ccNameTF.setEnabled(false);
      ccExpiryTF.setEnabled(false);
      linesList.setEnabled(false);
      deleteLineBtn.setEnabled(false);
      cdSearchBtn.setEnabled(false);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText("");
      newQtyTF.setEnabled(false);
      saveLineBtn.setEnabled(false);
   }

   /**
    * Disables/enables parts of the GUI for when the user is viewing
    * an Order.
    */
   private void setGUIForOrderViewing()
   {  newBtn.setEnabled(true);
      insertBtn.setEnabled(false);
      cancelBtn.setEnabled(false);
      modifyBtn.setEnabled(true);
      deleteBtn.setEnabled(true);
      orderList.setEnabled(true);
      displayCurrentOrder();
      salesPersonTF.setEnabled(true);
      dateTF.setEnabled(true);
      statusTF.setEnabled(true);
      recipientTF.setEnabled(true);
      shipHouseStreetTF.setEnabled(true);
      shipTownCountyTF.setEnabled(true);
      ccTypeTF.setEnabled(true);
      ccNumTF.setEnabled(true);
      ccNameTF.setEnabled(true);
      ccExpiryTF.setEnabled(true);
      linesList.setEnabled(true);
      deleteLineBtn.setEnabled(true);
      cdSearchBtn.setEnabled(true);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText("");
      newQtyTF.setEnabled(true);
      saveLineBtn.setEnabled(true);
   }

   /**
    * Disables/enables parts of the GUI for when the user is entering
    * a new Order.
    */
   private void setGUIForOrderEntry()
   {  newBtn.setEnabled(false);
      insertBtn.setEnabled(true);
      cancelBtn.setEnabled(true);
      modifyBtn.setEnabled(false);
      deleteBtn.setEnabled(false);
      orderList.setEnabled(false);
      currentOrder = null;
      displayCurrentOrder();
      salesPersonTF.setEnabled(true);
      dateTF.setEnabled(true);
      statusTF.setEnabled(true);
      recipientTF.setEnabled(true);
      shipHouseStreetTF.setEnabled(true);
      shipTownCountyTF.setEnabled(true);
      ccTypeTF.setEnabled(true);
      ccNumTF.setEnabled(true);
      ccNameTF.setEnabled(true);
      ccExpiryTF.setEnabled(true);
      linesList.setEnabled(false);
      deleteLineBtn.setEnabled(false);
      cdSearchBtn.setEnabled(false);
      newCDIdNumTF.setText("");
      newCD = null;
      newQtyTF.setText(""); 
      newQtyTF.setEnabled(false);
      saveLineBtn.setEnabled(false);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private ICDGUI parentComp;
   private OrderCollection orderCollection;
   private CDCollection cdCollection;

   private Order currentOrder;
   private Customer currentCustomer;
   private Vector viewedOrders;
   private CD newCD;

   private JList orderList;

   private JTextField custIdNumTF;
   private JTextField custNameTF;
   private JTextField custHouseStreetTF;
   private JTextField custTownCountyTF;
   private JTextField custTeleNumTF;
   private JTextField idNumTF;
   private JTextField salesPersonTF;
   private JTextField dateTF;
   private JTextField statusTF;
   private JTextField recipientTF;
   private JTextField shipHouseStreetTF;
   private JTextField shipTownCountyTF;
   private JTextField ccTypeTF;
   private JTextField ccNumTF;
   private JTextField ccNameTF;
   private JTextField ccExpiryTF;
   private JButton newBtn;
   private JButton insertBtn;
   private JButton cancelBtn;
   private JButton modifyBtn;
   private JButton deleteBtn;
   private JList linesList;
   private JButton deleteLineBtn;
   private JButton cdSearchBtn;
   private JTextField newCDIdNumTF;
   private JTextField newQtyTF;
   private JButton saveLineBtn;

   private JButton returnBtn;

   private static final int ID_TF_COLS = 6;
   private static final int NAME_TF_COLS = 30;
   private static final int ADDRESS_TF_COLS = 30;
   private static final int TELE_TF_COLS = 10;
   private static final int DATE_TF_COLS = 8;
   private static final int SALESPERSON_TF_COLS = 20;
   private static final int STATUS_TF_COLS = 20;
   private static final int CCTYPE_TF_COLS = 10;
   private static final int CCNUM_TF_COLS = 20;
   private static final int CDID_TF_COLS = 10;
   private static final int QTY_TF_COLS = 6;

   private static final String NEW = "NEW ORDER";
   private static final String INSERT = "INSERT";
   private static final String CANCEL = "CANCEL";
   private static final String MODIFY = "Save Changes";
   private static final String DELETE = "DELETE";
   private static final String DELETE_LINE = "Delete Line";
   private static final String FIND_CD = "Choose CD";
   private static final String SAVE_LINE = "Save New Order Line";
}
