import java.util.*;

/**
 * A class that represents reactive agents that inhabit grid-like worlds
 * and that are implemented using neural networks.
 * As written at the moment, it uses only my fully-connected, two layered,
 * feedforward nets.
 * @author Derek Bridge
 */
public abstract class NeuralNetAgent
   extends Agent
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new agent.
    * @param theNet the network that encodes this agent's action function.
    */
   public NeuralNetAgent(TwoLayerNet theNet)
   {  net = theNet;
      /* All agents have a full set of light, smell and touch sensors
         mounted around their bodies. Readings from these sensors go into
         arrays. But, so that we can experiment with different kinds of
         agents, these sensors can be enabled/disabled. Whether they
         are enabled or not is specified by an array of booleans.
       */
      hasLightSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      lightReading = new float[World.NUM_OF_COMPASS_POINTS];
      hasSmellSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      smellReading = new float[World.NUM_OF_COMPASS_POINTS];
      smellTypeReading = new int[World.NUM_OF_COMPASS_POINTS];
      hasTouchSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      touchReading = new boolean[World.NUM_OF_COMPASS_POINTS];
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */
  
   /**
    * Returns a string representation of this agent's net.
    */
   public String toString()
   {  return net.toString();
   }

/* --Setters----------------------------------------------------------- */

   /**
    * What to do on each clock tick.
    * @param theTime an integer representing how many ticks have passed.
    */
   public void onTick(int theTime)
   {  sense();
      double[] s = createNetInputs();
      double[] output = net.activate(s);
      Action action = decodeNetOutputs(output);
      action.execute(this);
   }

   /**
    * Take sensor readings.
    */
   public void sense()
   {  Cell neighbour = null;
      int sens = FRONT;
      int dir = direction;
      for (int i = 0; i < World.NUM_OF_COMPASS_POINTS; 
         i++, sens = (sens + 1) % World.NUM_OF_COMPASS_POINTS, 
         dir = World.spin(dir, 1))
      {  neighbour = cell.getNeighbour(dir);
         if (hasLightSensor[sens])
         {  lightReading[sens] = neighbour.getAmountOfLight();
         }
         if (hasSmellSensor[sens])
         {  smellReading[sens] = neighbour.getAmountOfSmell();
            smellTypeReading[sens] = neighbour.getSmellType();
         }
         if (hasTouchSensor[sens])
         {  touchReading[sens] = neighbour.someSpaceIsInUse();
         }
      }
   }

   /**
    * Forms an array of doubles from the sensor readings for
    * input to the net. This is agent-specific, so here it is an
    * abstract method.
    * @return the array of inputs to the net.
    */
   public abstract double[] createNetInputs();

   /**
    * Takes the output of the net and works out which action
    * this signifies. This is agent-specific, so here it is an
    * abstract method.
    * @param theNetOutput the output of the net.
    * @return the action.
    */
   public abstract Action decodeNetOutputs(double[] theNetOutput);

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected TwoLayerNet net;

   /* An agent can have a number of light/smell/touch sensors at various
      positions around its perimeter. The positions are give by named
      constants above, e.g. FRONT_OF_HEAD. 
      Arrays of booleans say whether they are enabled or not. 
      They must be explicitly enabled, and this would be done in subclasses.
      Their readings are kept in another array. Light and smell readings are 
      real numbers; but touch sensors simply return true or false;
    */
   protected boolean[] hasLightSensor;
   protected float[] lightReading;

   protected boolean[] hasSmellSensor;
   protected float[] smellReading;
   protected int[] smellTypeReading;

   protected boolean[] hasTouchSensor;
   protected boolean[] touchReading;
}
