import java.io.*;
import java.util.*;

/**
 * A class that implements some handy methods for writing demos of
 * learning algorithms.
 * @author Derek Bridge
 */
public class LearningUtilities
{
   /**
    * Read in a data set from a file.
    * Assumptions: the file is a text file, each example is one line;
    * values within examples are comma-separated; all values
    * in the file can be stored as type double.
    * @param theFileName the name of the file containing the data.
    * @param theNumOfInputs the number of values that comprise each
    * exmaple input.
    * @param theNumOfOutputs the number of values that comprise each
    * example output.
    */
   public static Map readExamples(String theFileName, 
      int theNumOfInputs, int theNumOfOutputs)
   {  Map examples = new Hashtable();
      double[] exampleInput;
      double[] targetOutput;
      BufferedReader stream = null;
      try
      {  /* Open stream
          */
         stream = new BufferedReader(new FileReader(theFileName));
         /* Read each String from the file and tokenise it into
            (splitting at the commas) into two arrays of doubles.
          */
         boolean isEOF = false;
         do
         {  String nextString = stream.readLine(); 
            if (nextString == null)
            {  isEOF = true;
            }
            else
            {  exampleInput = new double[theNumOfInputs];
               targetOutput = new double[theNumOfOutputs];
               readExample(nextString, 
                  theNumOfInputs, exampleInput,
                  theNumOfOutputs, targetOutput);
               examples.put(exampleInput, targetOutput);               
            }
         } while (! isEOF);
      }
      catch (IOException ioe)
      {  System.out.println("Error: problem reading your data file");
      }
      finally
      {  try
         {  if (stream != null)
            {  stream.close();
            }
         }
         catch (IOException ioe)
         {  System.out.println("Error: problem reading your data file");
         }
         return examples;
      }
   }

   /**
    * Splits a data set into two: a training set and a test set.
    * (A somewhat brute force approach!)
    * @param theDataSet the full set of examples.
    * @param theTrainingPerCent what percentage should be training data.
    * @param trainingSet the map into which we'll place the training data.
    * @param testSet the map into which we'll place the test data.
    */
   public static void splitData(Map theDataSet, int theTrainingPerCent,
      Map trainingSet, Map testSet)
   {  Object[] a = theDataSet.entrySet().toArray();
      int size = a.length;
      for (int i = 0; i < size; i++)
      {  int j = (int)(Math.random() * size);
         Object temp = a[j];
         a[j] = a[i];
         a[i] = temp;
      }
      int trainingSize = (int) (size * theTrainingPerCent / 100);
      int testSize = size - trainingSize;
      for (int i = 0; i < trainingSize; i++)
      {  Map.Entry me = (Map.Entry) a[i];
         trainingSet.put(me.getKey(), me.getValue());
      }
      for (int i = 0; i < testSize; i++)
      {  Map.Entry me = (Map.Entry) a[i];
         testSet.put(me.getKey(), me.getValue());
      }
  }

   /**
    * Extract example inputs and target outputs from a string.
    * @param str the string we're extracting from.
    * @param theNumOfInputs how many input values there should be.
    * @param exampleInputs the array into which we'll place the inputs.
    * @param theNumOfOutputs how many outputs there should be.
    * @param exampleOutputs the array into which we'll place the outputs.
    */
   private static void readExample(String str, 
      int theNumOfInputs, double[] exampleInputs,
      int theNumOfOutputs, double[] exampleOutputs)
   {  try
      {  StringTokenizer tokenizer = new StringTokenizer(str, ",");
         for (int i = 0; i < theNumOfInputs; i++)
         {  String token = tokenizer.nextToken();
            exampleInputs[i] = Double.parseDouble(token);
         }      
         for (int i = 0; i < theNumOfOutputs; i++)
         {  String token = tokenizer.nextToken();
            exampleOutputs[i] = Double.parseDouble(token);
         }
      }
      catch (NoSuchElementException nsee)
      {  System.out.println("Error: file of examples not in right format");
         /* This isn't necessarily a good thing to do next...
          */
         System.exit(1);
      }
      catch (NumberFormatException nfe)
      {  System.out.println("Error: file of examples not in right format");
         /* This isn't necessarily a good thing to do next...
          */
         System.exit(1);
      }
   }   
}
