package ga;
import java.util.*;
import world.*;
import proplogic.*;
import actions.*;


/**
 * An abstract class that represents reactive agents implemented using 
 * production systems but who carry a representation of their production 
 * system as a bit-string to allow the production of offfspring by genetic
 * manipulation.
 * @author Derek Bridge
 */
public abstract class GAProductionSystemAgent
   extends ProductionSystemAgent
   implements IGAAgent
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new agent with a random set of genes.
    * @param theNumofBitsNeededToEncodeTheCondition this should be how ever
    * many different propositional symbols there are.
    * @param theNumOfBitsNeededToEncodeTheActions this should be
    * the smallest integer greater than or equal to the log to the base 2
    * of the length of theActions. I ask you to pass this in (rather than
    * calculate it myself) because the standard Java API doesn't offer a
    * method for computing anything other than natural logs.
    * @param theNumOfRules the maximum number of rules allowed in the
    * production system of this agent.
    */
   public GAProductionSystemAgent(int theNumOfBitsNeededToEncodeTheCondition, 
      int theNumOfBitsNeededToEncodeTheActions, int theNumOfRules)
   {  numOfPropBits = theNumOfBitsNeededToEncodeTheCondition;
      numOfActionBits = theNumOfBitsNeededToEncodeTheActions;
      numOfRules = theNumOfRules;
      numOfBitsPerRule = numOfPropBits + numOfActionBits;
      genes = new String(generateRandomGenes());
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns this agent's fitness.
    */
   public float getFitness()
   {  return fitness;
   }

   /**
    * Returns this agent's genes
    */
   public String getGenes()
   {  return genes;
   }

   /**
    * Create offspring by copying this agent.
    */
   public Object clone()
   {  try
      {  GAProductionSystemAgent agent = (GAProductionSystemAgent) super.clone();
         return agent;
      }
      catch (CloneNotSupportedException cnse)
      {  return null; // cannot happen
      }
   }

   /**
    * Create offspring by mutation of genes.
    */
   public IGAAgent createMutatedChild()
   {  int randPosn = (int) (Math.random() * genes.length());
      char original = genes.charAt(randPosn);
      int randSwitch = (int) (Math.random() * 2);
      char newBit =
         original == '0' && randSwitch == 0 ? '1' :
         original == '0' && randSwitch == 1 ? '2' :
         original == '1' && randSwitch == 0 ? '0' :
         original == '1' && randSwitch == 1 ? '2' :
         original == '2' && randSwitch == 0 ? '0' : '1';
      String newGenes =
         genes.substring(0, randPosn) + newBit + 
         genes.substring(randPosn + 1, genes.length());
      GAProductionSystemAgent child = (GAProductionSystemAgent) clone();
      child.setGenes(newGenes);
      return child;
   }     

   /**
    * Create two offspring by combining genes from this agent and
    * another agent.
    * @param theMate the other parent.
    */
   public List createCombinedChildren(IGAAgent theMate)
   {  int randPosn = (int) (Math.random() * genes.length());
      String otherGenes = theMate.getGenes();
      String child1Genes = genes.substring(0, randPosn) + 
         otherGenes.substring(randPosn, otherGenes.length());
      String child2Genes = otherGenes.substring(0, randPosn) + 
         genes.substring(randPosn, genes.length());
      GAProductionSystemAgent child1 = (GAProductionSystemAgent) clone();
      child1.setGenes(child1Genes);
      GAProductionSystemAgent child2 = (GAProductionSystemAgent) clone();
      child2.setGenes(child2Genes);
      List children = new ArrayList(2);
      children.add(child1);
      children.add(child2);
      return children;
   }
    
/* --Setters----------------------------------------------------------- */

   /**
    * Provides the information needed to allow genes to be decoded.
    * @param theProps an array of the different propositional symbols that
    * can be used in the production system of this agent.
    * @param theActions an array of the different actions that can be used
    * in the production system of this agent.
    */
   public void setDecodeInfo(String[] theProps, Action[] theActions)
   {  props = theProps;
      actions = theActions;
      rules = decodeGenes();
   }  

   /**
    * Sets this agent's fitness.
    * @param theFitness the agent's fitness.
    */
   public void setFitness(float theFitness)
   {  fitness = theFitness;
   }

   /** 
    * Sets this agent's genes.
    * @param theGenes the agent's genes.
    */
   public void setGenes(String theGenes)
   {  genes = theGenes;
      rules = decodeGenes();
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Creates this agent's production system by decoding its genes.
    */
   private ProductionSystem decodeGenes()
   {  ProductionSystem rules = new ProductionSystem();
      // Create the rules by decoding the genes.
      for (int i = 0; i < numOfRules; i++)
      {  // For this rule, first create its condition.
         List literals = new ArrayList(numOfPropBits);
         for (int j = 0; j < numOfPropBits; j++)
         {  char bit = genes.charAt(i * numOfBitsPerRule + j);
            if (bit == '0')
            {  literals.add(new NOT(new Proposition(props[j])));
            }
            else if (bit == '1')
            {  literals.add(new Proposition(props[j]));
            }
         }
         BooleanExpression condition =
            createConjoinedExpression(literals);
         // Then create its action.
         String actionGenes = genes.substring(i * numOfBitsPerRule + numOfPropBits, 
            i * numOfBitsPerRule + numOfBitsPerRule);
         int actionId = -1;
         try
         {  actionId = Integer.parseInt(actionGenes, 2);
         }
         catch (NumberFormatException nfe)
         {  // This might happen if a condition bit (for which only 0 or 1
            // is allowed) got set to 2 during evolution.
            // We leave actionId at -1 and handle this below.
         }
         if (actionId == -1 || actionId >= actions.length)
         {  actionId = (int) (Math.random() * actions.length);
         }
         Action action = actions[actionId];
         // So now create the rule.
         StimulusResponseRule rule = 
            new StimulusResponseRule(condition, action);
         rules.addRule(rule);
      }
      // Always add this default rule so that execution of the agent cannot fail.
      rules.addRule(
         new StimulusResponseRule(new True(), new NullAction()));
      return rules;
   }

   /**
    * Conjoins together a number of literals.
    * @param theLiterals the literals to be conjoined.
    */
   private BooleanExpression createConjoinedExpression(List theLiterals)
   {  if (theLiterals.isEmpty())
      {  return new True();
      }
      if (theLiterals.size() == 1)
      {  return (BooleanExpression) theLiterals.get(0);
      }
      return new AND((BooleanExpression) theLiterals.get(0),
         createConjoinedExpression(
            theLiterals.subList(1, theLiterals.size())));
   }

   /**
    * Generates a random bit string for agents in the very first 
    * population.
    */
   private char[] generateRandomGenes()
   {  char[] bits = new char[numOfRules * numOfBitsPerRule];
      // Generate each rule
      for (int i = 0; i < numOfRules; i++)
      {  // For this rule, first generate its condition bits
         for (int j = 0; j < numOfPropBits; j++)
         {  // these `bits' can take on values 0, 1 or 2
            bits[i * numOfBitsPerRule + j] = 
               Character.forDigit(((int) (Math.random() * 3)), 10);
         }
         // Then generate its action bits
         for (int k = 0; k < numOfActionBits; k++)
         {  // these bits can take values 0 or 1
            bits[i * numOfBitsPerRule + numOfPropBits + k] =
               Character.forDigit(((int) (Math.random() * 2)), 10);
         }
      }
      return bits;
   }     

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private String genes;
   private float fitness;
   private int numOfRules;
   private int numOfPropBits;
   private int numOfActionBits;
   private int numOfBitsPerRule;
   private String[] props;
   private Action[] actions;
}
