package actions;
import world.*;

/**
 * A class that represents move actions by agents that inhabit grid-like worlds.
 * @author Derek Bridge
 */
public class Move
   extends Action
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new move action, where the number of cells moved is 1.  
    */
   public Move()
   {  numOfSteps = 1;
   }

   /**
    * Allocates a new move action, where the number of cells moved is given.
    * @param theNumOfSteps the number of steps to take.
    */
   public Move(int theNumOfSteps)
   {  numOfSteps = theNumOfSteps;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns true if moving the given number of steps in the given
    * direction caused a crash.
    */
   public boolean didCrash()
   {  return didCrash;
   }

   /**
    * Returns a string representation of this expression.
    */
   public String toString()
   {  return numOfSteps == 1 ? "move" : "move" + numOfSteps;
   }


/* --Setters----------------------------------------------------------- */

   /**
    * Carry out the action in the world.
    * @param theAgent the agent who is executing this action.
    */
   public void execute(Agent theAgent)
   {  Cell currentCell = theAgent.getCell();
      int direction = theAgent.getDirection();
      Cell nextCell = null;
      didCrash = false;
      boolean hasRoomToMove = false;
      int diagDirection1 = 0;
      int diagDirection2 = 0;
      for (int i = 0; i <  numOfSteps; i++)
      {  nextCell = currentCell.getNeighbour(direction);
         if (direction == World.NORTH || direction == World.EAST ||
            direction == World.SOUTH || direction == World.WEST)
         {  hasRoomToMove = nextCell.canAccommodate(theAgent);
         }
         else
         {  diagDirection1 = World.spin(direction, 1);
            diagDirection2 = World.spin(direction, -1);
            hasRoomToMove = nextCell.canAccommodate(theAgent) &&
               currentCell.getNeighbour(diagDirection1).canAccommodate(theAgent) &&
               currentCell.getNeighbour(diagDirection2).canAccommodate(theAgent);
         }
         if (hasRoomToMove)
         {  currentCell.removeObject(theAgent);
            nextCell.addObject(theAgent);
            theAgent.setCell(nextCell);
            currentCell = nextCell;
         }
         else
         {  // Can't move; so do nothing
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private int numOfSteps;
   private boolean didCrash;
}
