package world;
import java.util.*;
import java.awt.*;
import javax.swing.*;

/**
 * A class that represents grid-like worlds.
 * For efficiency, worlds and their graphical representations (and cells
 * and their graphical representations) are collapsed into a single
 * class (whereas, for flexibility, PhysObjects and their graphical
 * representations are handled by separate classes).
 * @author Derek Bridge
 */
public class World
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new world with given dimensions.
    * @param theWidth the width of the world.
    * @param theHeight the height of the world.
    * @param theScaleFactor the scale factor used for displaying the world.
    */
   public World(int theWidth, int theHeight, int theScaleFactor)
   {  width = theWidth;
      height = theHeight;
      scaleFactor = theScaleFactor;
      setPreferredSize(new Dimension(width * scaleFactor, height * scaleFactor));
      createCells();
      objects = createObjects();
      numOfSmellTypes = DEFAULT_NUM_OF_SMELL_TYPES;
      lightFallOffRate = DEFAULT_LIGHT_FALL_OFF_RATE;
      backgroundColour = DEFAULT_BACKGROUND_COLOUR;
      gridLineColour = DEFAULT_GRIDLINE_COLOUR;
      colourOfLight = DEFAULT_COLOUR_OF_LIGHT;
      colourOfSmells = new Color[numOfSmellTypes];
      for (int i = 0; i < colourOfSmells.length; i++)
      {  colourOfSmells[i] = DEFAULT_COLOUR_OF_SMELLS[i];
      }
   }
      
/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Public class constants-------------------------------------------- */

   public static final int EAST = 0;
   public static final int NORTHEAST = 1;
   public static final int NORTH = 2;
   public static final int NORTHWEST = 3;
   public static final int WEST = 4;
   public static final int SOUTHWEST = 5;
   public static final int SOUTH = 6;
   public static final int SOUTHEAST = 7;

   public static final int NUM_OF_COMPASS_POINTS = 8;

/* --Getters----------------------------------------------------------- */

   /**
    * Returns the width of this world.
    */
   public int getWorldWidth()
   {  return width;
   }

   /**
    * Returns the height of this world.
    */
   public int getWorldHeight()
   {  return height;
   }

   /**
    * Returns the scale factor of this world.
    */
   public int getScaleFactor()
   {  return scaleFactor;
   }

   /**
    * Returns the world's cells.
    */
   public Cell[][] getCells()
   {  return cells;
   }

   /**
    * Returns the world's objects.
    */
   public java.util.List getObjects()
   {  return objects;
   }

   /**
    * Returns the light fall off rate.
    */
   public float getLightFallOffRate()
   {  return lightFallOffRate;
   }

   /**
    * Returns the smell fall off rate for a given type of smell.
    * @param theSmellType the type of smell whose fall off rate is sought.
    */
   public float getSmellFallOffRate(int theSmellType)
   {  return smellFallOffRates[theSmellType];
   }

   /**
    * Returns the smell evaporation rate for a given type of smell.
    * @param theSmellType the type of smell whose evaporation rate is sought.
    */
   public float getSmellEvaporationRate(int theSmellType)
   {  return smellEvaporationRates[theSmellType];
   }

   /**
    * Returns the number of different types of smell this world can reek of.
    */
   public int getNumOfSmellTypes()
   {  return numOfSmellTypes;
   }

   /**
    * Returns the background colour.
    */
   public Color getBackgroundColour()
   {  return backgroundColour;
   }

   /**
    * Returns the colour of gridlines.
    */
   public Color getGridLineColour()
   {  return gridLineColour;
   }

   /**
    * Returns whether gridlines are visible or not.
    */
   public boolean gridLinesAreVisible()
   {  return gridLinesAreVisible;
   }
   /**
    * Returns the colour of light.
    */
   public Color getColourOfLight()
   {  return colourOfLight;
   }

   /**
    * Returns whether light is visible or not.
    */
   public boolean lightIsVisible()
   {  return lightIsVisible;
   }

   /**
    * Returns the colour of a particular type of smell.
    * @param theSmellType the type of smell whose colour being sought.
    */
   public Color getColourOfSmell(int theSmellType)
   {  return colourOfSmells[theSmellType];
   }

   /**
    * Returns whether smell is visible or not.
    */
   public boolean smellIsVisible()
   {  return smellIsVisible;
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Creates physical objects and places them in the world.
    */
   public java.util.List createObjects()
   {  return new LinkedList();
   }

   /**
    * What to do on each clock tick.
    * @param theTime an integer representing how many ticks have passed.
    */
   public void onTick(int theTime)
   {  for (int i = 0; i < width; i++)
      {  for (int j = 0; j < height; j++)
         {  cells[i][j].onTick(theTime);
         }
      }
      Iterator objectsIterator = objects.iterator();
      while (objectsIterator.hasNext())
      {  PhysObject object = (PhysObject) objectsIterator.next();
         object.onTick(theTime);
      }
   }

   /**
    * Sets the fall off rate for light.
    * @param theFallOffRate the rate by which light falls off
    * as it spreads.
    */
   public void setLightFallOffRate(float theFallOffRate)
   {  lightFallOffRate = theFallOffRate;
   }

   /**
    * Sets up the smells that this world accommodates.
    * The arrays must all have the same length. This is not checked here!
    * @param theSmellFallOffRates the rate by which smell falls off.
    * @param theSmellEvaportaionRates the rate by which smells evaporate.
    * @param theSmellColours the colours of the smells.
    */
   public void setSmells(float[] theSmellFallOffRates,
      float[] theSmellEvaporationRates, Color[] theSmellColours)
   {  numOfSmellTypes = theSmellFallOffRates.length;
      smellFallOffRates = new float[numOfSmellTypes];
      smellEvaporationRates = new float[numOfSmellTypes];
      colourOfSmells = new Color[numOfSmellTypes];
      for (int i = 0; i < numOfSmellTypes; i++)
      {  smellFallOffRates[i] = theSmellFallOffRates[i];
         smellEvaporationRates[i] = theSmellEvaporationRates[i];
         colourOfSmells[i] = theSmellColours[i];
      }
   }

   /**
    * Sets the fall off rate for the given smell type.
    * @param theSmellType the type of smell whose fall off rate
    * is being set (must be < numOfSmellTypes).
    * @param theFallOffRate the rate by which this smell falls off
    * as it spreads.
    */
   public void setSmellFallOffRate(int theSmellType, float theFallOffRate)
   {  smellFallOffRates[theSmellType] = theFallOffRate;
   }

   /**
    * Sets the evaporation rate for the given smell type.
    * @param theSmellType the type of smell whose evaporation rate
    * is being set (must be < numOfSmellTypes).
    * @param theEvaporationRate the rate by which this smell evaporates
    * as time passes.
    */
   public void setSmellEvaporationRate(int theSmellType, float theEvaporationRate)
   {  smellEvaporationRates[theSmellType] = theEvaporationRate;
   }

   /**
    * Add an object to the world.
    * @param anObject the object being added.
    */
   public void addObject(PhysObject anObject)
   {  objects.add(anObject);
   }

   /**
    * Remove an object from the world.
    * @param anObject the object being removed.
    */
   public void removeObject(PhysObject anObject)
   {  objects.remove(anObject);
   }

   /**
    * Returns a compass point based on a given compass point and a number
    * of positions to be added to that point.
    * @param startPoint the initial compass point.
    * @param addOn the positions to add.
    * The addOn shouldn't be too big (< NUM_OF_COMPASS_POINTS). 
    * (In fact the code works OK for any +ve addOn. But -ve addOns 
    * mustn't be less than - NUM_OF_COMPASS_POINTS.)
    */
   public static int spin(int startPoint, int addOn)
   {  int newPoint = startPoint + addOn;
      if (newPoint >= NUM_OF_COMPASS_POINTS)
      {  return newPoint % World.NUM_OF_COMPASS_POINTS;
      }
      else if (newPoint < 0)
      {  return newPoint + NUM_OF_COMPASS_POINTS;
      }
      else
      {  return newPoint;
      }
   }

   /**
    * Sets the background colour.
    * @param theColour the new background colour.
    */
   public void setBackgroundColour(Color theColour)
   {  backgroundColour = theColour;
   }

   /**
    * Sets the gridline colour.
    * @param theColour the new gridline colour.
    */
   public void setGridLineColour(Color theColour)
   {  gridLineColour = theColour;
   }

   /**
    * Sets the gridlines to visible.
    * @param theStatus whether gridlines are visible or not.
    */
   public void setGridLinesVisible(boolean theStatus)
   {  gridLinesAreVisible = theStatus;
   }

   /**
    * Sets the colour of light.
    * @param theColour the new colour of light.
    */
   public void setColourOfLight(Color theColour)
   {  colourOfLight = theColour;
   }

   /**
    * Sets the light to visible.
    * @param theStatus whether light is visible or not.
    */
   public void setLightVisible(boolean theStatus)
   {  lightIsVisible = theStatus;
   }

   /**
    * Sets the colour of a given type of smell.
    * @param theSmellType the type of smell whose colour is being set.
    * @param theColour the new colour for this type of smell.
    */
   public void setColourOfSmell(int theSmellType, Color theColour)
   {  colourOfSmells[theSmellType] = theColour;
   }

   /**
    * Sets the smell to visible.
    * @param theStatus whether smell is visible or not.
    */
   public void setSmellVisible(boolean theStatus)
   {  smellIsVisible = theStatus;
   }

   /**
    * Repaint this panel.
    * @param g the graphics component.
    */
   public void paintComponent(Graphics g)
   {  Graphics2D g2d = (Graphics2D) g;
      g2d.scale(scaleFactor, scaleFactor);
      g2d.setStroke(new BasicStroke(2.0f / scaleFactor));
      super.paintComponent(g2d);
      for (int i = 0; i < width; i++)
      {  for (int j = 0; j < height; j++)
         {  cells[i][j].draw(g2d);
         }
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Create and link the cells in this world.
    */
   private void createCells()
   {  cells = new Cell[width][height];
      for (int i = 0; i < width; i++)
      {  for (int j = 0; j < height; j++)
         {  cells[i][j] = new Cell(this, i, j);
         }
      }
      for (int i = 0; i < width; i++)
      {  for (int j = 0; j < height; j++)
         {  Cell c = cells[i][j];
            int n = j - 1;
            int e = i + 1;
            int s = j + 1;
            int w = i - 1;
            if (n >= 0)
            {  c.setNeighbour(NORTH, cells[i][n]);
               if (e < width)
               {  c.setNeighbour(NORTHEAST, cells[e][n]);
               }
               if (w >= 0)
               {  c.setNeighbour(NORTHWEST, cells[w][n]);
               }
            }
            if (s < height)
            {  c.setNeighbour(SOUTH, cells[i][s]);
               if (e < width)
               {  c.setNeighbour(SOUTHEAST, cells[e][s]);
               }
               if (w >= 0)
               {  c.setNeighbour(SOUTHWEST, cells[w][s]);
               }
            }
            if (e < width)
            {  c.setNeighbour(EAST, cells[e][j]);
            }
            if (w >= 0)
            {  c.setNeighbour(WEST, cells[w][j]);
            }
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected int width;
   protected int height;
   protected Cell[][] cells; 
   protected java.util.List objects;
   protected float lightFallOffRate;
   protected int numOfSmellTypes;
   protected float[] smellFallOffRates;
   protected float[] smellEvaporationRates;

   protected Color backgroundColour;
   protected Color gridLineColour;
   protected Color colourOfLight;
   protected Color[] colourOfSmells;
   protected boolean gridLinesAreVisible;
   protected boolean smellIsVisible;
   protected boolean lightIsVisible;
   protected int scaleFactor;

   private static final int DEFAULT_NUM_OF_SMELL_TYPES = 4;
   private static final float DEFAULT_LIGHT_FALL_OFF_RATE = 0.2f;
   private static final float DEFAULT_SMELL_FALL_OFF_RATE = 0.2f;
   private static final float DEFAULT_SMELL_EVAPORATION_RATE = 0.2f;

   private static final Color DEFAULT_BACKGROUND_COLOUR = Color.white;
   private static final Color DEFAULT_GRIDLINE_COLOUR = Color.lightGray;
   private static final Color DEFAULT_COLOUR_OF_LIGHT = Color.yellow;
   private static final Color[] DEFAULT_COLOUR_OF_SMELLS = 
      {Color.magenta, Color.cyan, Color.green, Color.pink};
}
