package nets;

/**
 * A class that represents fully-connected, multilayered, feedforward
 * networks of threshold logic units (TLUs), where there are only two 
 * layers.
 * @author Derek Bridge
 */
public class TwoLayerNet
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new net with the given numbers of units.
    * (These parameters exclude the `extras' needed to replace
    * thresholds by extra units.)
    * @param theNumOfInputs the number of inputs.
    * @param theNumOfHiddenUnits the number of hidden units.
    * @param theNumOfOutputs the number of output units.
    */
   public TwoLayerNet(int theNumOfInputs, int theNumOfHiddenUnits,
      int theNumOfOutputs)
   {  numOfInputs = theNumOfInputs + 1;
      numOfHiddenUnits = theNumOfHiddenUnits + 1;
      hiddenLayer = new HiddenLayerTLU[numOfHiddenUnits];
      hiddenLayer[0] = new DummyTLU();
      for (int i = 1; i < numOfHiddenUnits; i++)
      {  hiddenLayer[i] = 
            new HiddenLayerTLU(numOfInputs, 
               getRandomWeightVector(numOfInputs));
      }
      numOfOutputs = theNumOfOutputs;
      outputLayer = new OutputLayerTLU[numOfOutputs];
      for (int i = 0; i < numOfOutputs; i++)
      {  outputLayer[i] = 
            new OutputLayerTLU(numOfHiddenUnits, hiddenLayer,
               getRandomWeightVector(numOfHiddenUnits));
      }
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Computes the activation (outputs) of this net, given a vector of
    * of sensor values.
    * @param s an array of sensor values (the input to this net).
    * (The length of this array is whatever value the user supplied as 
    * theNumOfInputs in the constructor. It should exclude the extra 
    * input that results from treating the threshold as a weighted 
    * input. This extra input is handled directly in the code.)
    */
   public double[] activate(double[] s)
   {  /* The hidden layer is activated directly from the sensor values.
       */
      for (int i = 0; i < numOfHiddenUnits; i++)
      {  hiddenLayer[i].activate(s);
      }
      /* The output layer is activated from the outputs of the hidden
         layer (hence we call activate() instead of activate(s)).
         And the outputs of the net as a whole are the outputs of the
         TLUs in this output layer.
       */ 
      double[] outputs = new double[numOfOutputs];
      for (int i = 0; i < numOfOutputs; i++)
      {  outputs[i] = outputLayer[i].activate();
      }
      return outputs;
   }

   /**
    * Returns the jth TLU in the net's hidden layer.
    * @param j an index for the hidden unit. 
    * (The index j can range from 0 to numOfHiddenUnits
    * inclusive. Line 0 is the extra line that replaces 
    * the threshold.)
    */
   public TLU getHiddenUnit(int j)
   {  return hiddenLayer[j];
   }

   /**
    * Returns the kth TLU in the net's output layer.
    * @param k an index for the hidden unit.
    * (The index k can range from 0 to numOfOutputUnits - 1
    * inclusive. There is no extra unit in the output layer.)
    */
   public TLU getOutputUnit(int k)
   {  return outputLayer[k];
   }

   /**
    * Returns the weight on a line between a sensor input and a hidden
    * unit.
    * @param i an index for the sensor input. (The index i ranges
    * from 0 to theNumOfInputs inclusive, where 0 is the extra input.)
    * @param j an index for the hidden unit. (The index j ranges
    * from 1 to theNumOfHiddenUnits inclusive. There is a hidden unit
    * indexed by j = 0, but it takes no inputs.)
   */
   public double getHiddenLayerWeight(int i, int j)
   {  return hiddenLayer[j].getWeight(i);
   }

   /**
    * Returns the weight on a line between a hidden unit and an output
    * unit.
    * @param j an index for the hidden unit. (The index j ranges
    * from 0 to theNumOfHiddenUnits, where 0 is the extra unit.)
    * @param k an index for the output unit. (The index k ranges
    * from 0 to theNumOfOutputUnits - 1. There is no extra unit.)
    */
   public double getOutputLayerWeight(int j, int k)
   {  return outputLayer[k].getWeight(j);
   }

   /**
    * Returns a string representation of this net.
    */
   public String toString()
   {  StringBuffer sb = new StringBuffer();
      sb.append("Two Layer Net\n=============\n\n");
      sb.append("   Input Units to Hidden Units\n" +
         "   ---------------------------\n");
      for (int i = 0; i < numOfInputs; i++)
      {  sb.append("      Input unit " + i + "\n");
         for (int j = 1; j < numOfHiddenUnits; j++)
         {  sb.append("      ---weight " + 
               getHiddenLayerWeight(i, j) + 
               "---Hidden unit " + j + "\n");
         }
      }
      sb.append("\n   Hidden Units to Output Units\n" +
         "   ----------------------------\n");
      for (int j = 0; j < numOfHiddenUnits; j++)
      {  sb.append("      Hidden unit " + j + "\n");
         for (int k = 0; k < numOfOutputs; k++)
         {  sb.append("      ---weight " + 
               getOutputLayerWeight(j, k) + 
               "---Output unit " + k + "\n");
         }
      }
      return sb.toString();
   }

   /**
    * Returns a random threshold for a TLU within this net.
    * (This (negated) becomes the weight on an extra input line.)
    */
   public static double getRandomThreshold()
   {  return Math.random() - 0.5;
   }

   /**
    * Returns a vector of random weights for the input lines of a TLU
    * within this net.
    * @param n the size of the desired vector.
    */
   public static double[] getRandomWeightVector(int n)
   {  double[] weights = new double[n];
      for (int i = 0; i < n; i++)
      {  weights[i] = Math.random() - 0.5;
      }
      return weights;
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Adjust the weight on a line between a sensor input and a hidden
    * unit.
    * @param i an index for the sensor input. (The index i ranges
    * from 0 to theNumOfInputs inclusive, where 0 is the extra input.)
    * @param j an index for the hidden unit. (The index j ranges
    * from 1 to theNumOfHiddenUnits inclusive. There is an extra
    * hidden unit but it takes no inputs.)
    * @param theWeight the new weight.
    */
   public void setHiddenLayerWeight(int i, int j, double theWeight)
   {  hiddenLayer[j].setWeight(i, theWeight);
   }

   /**
    * Adjust the weight on a line between a hidden unit and an output
    * unit.
    * @param j an index for the hidden unit. (The index j ranges
    * from 0 to theNumOfHiddenUnits inclusive, where 0 is the extra 
    * unit.)
    * @param k an index for the output unit. (The index k ranges
    * from 0 to theNumOfOutputUnits - 1. There is no extra unit in
    * the output layer.)
    * @param theWeight the new weight.
    */
   public void setOutputLayerWeight(int j, int k, double theWeight)
   {  outputLayer[k].setWeight(j, theWeight);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected int numOfInputs;

   protected int numOfHiddenUnits;
   protected HiddenLayerTLU[] hiddenLayer;

   protected int numOfOutputs;
   protected OutputLayerTLU[] outputLayer;
}
