package gui;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import world.*;

/**
 * A class that represents controllers that respond to events in a grid-like world.
 * @author Derek Bridge
 */
public class WorldController
   implements MouseListener, ActionListener, MenuListener
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new controller.
    * @param theWorld the world being controlled.
    */
   public WorldController(World theWorld)
   {  frame = new JFrame("Agent Demonstrator");
      frame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  actionPerformed(
               new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "exit"));
         }
      });
      world = theWorld;
      world.addMouseListener(this);
      frame.getContentPane().add(world);
      menuBar = new WorldMenuBar(frame, world, this, this);
      frame.setJMenuBar(menuBar);
      frame.pack();
      frame.setLocation(200, 200);
      frame.setVisible(true);
      time = 0;
      interval = DEFAULT_INTERVAL;
      clock =
         new javax.swing.Timer(DEFAULT_INTERVAL, new ActionListener()
         {  public void actionPerformed(ActionEvent ae)
            {  world.onTick(time);
               world.repaint();
               time++;
            }
         });
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Sets the world for this controller.
    * @param theWorld the new world.
    */
   public void setWorld(World theWorld)
   {  frame.getContentPane().remove(world);
      world = theWorld;
      frame.getContentPane().add(world);
      frame.pack();
      world.repaint();
   }

   /**
    * Gets the timer interval.
    * @return the interval.
    */
    public synchronized int getInterval()
    {  return interval;
    }

   /**
    * Sets the timer interval.
    * @param theInterval the new interval.
    */
   public synchronized void setInterval(int theInterval)
   {  interval = theInterval;
      clock.setDelay(theInterval);
   }

   /**
    * Starts the clock.
    */
   public synchronized void startClock()
   {  if (! clock.isRunning())
      {  menuBar.disableOptions();
         clock.start();
      }
   }

   /**
    * Stops the clock.
    */
   public synchronized void stopClock()
   {  if (clock.isRunning())
      {  menuBar.enableOptions();
         clock.stop();
      }
   }

   /**
    * Menu item selected.
    */
   public void actionPerformed(ActionEvent ae)
   {  String command = ae.getActionCommand();
      if (command.equals("new"))
      {  doNew();
      }
      else if (command.equals("open"))
      {  doOpen();
      }
      else if (command.equals("save"))
      {  doSave();
      }
      else if (command.equals("exit"))
      {  doExit();
      }
      else if (command.equals("start"))
      {  startClock();
      }
      else if (command.equals("stop"))
      {  stopClock();
      }
      else if (command.equals("speed"))
      {  if (speedDialog == null)
         {  speedDialog = new SpeedDialog(frame, this);
         }
         speedDialog.setVisible(true);
      }
      else if (command.equals("gridLines"))
      {  doGridLines();
      }
      else if (command.equals("smell"))
      {  doSmell();
      }
      else if (command.equals("light"))
      {  doLight();
      }
      else if (command.equals("about"))
      {  doAbout();
      }
   }

   /**
    * Menu abandoned.
    */
   public void menuCanceled(MenuEvent me)
   {
   }

   /**
    * Menu un-posted after action.
    */
   public void menuDeselected(MenuEvent me)
   {
   }

   /**
    * Menu posted.
    */
   public void menuSelected(MenuEvent me)
   {
   }

   /**
    * Mouse clicked.
    */
   public void mouseClicked(MouseEvent me)
   {  showPopUp(me);
   }

   /**
    * Mouse entered.
    */
   public void mouseEntered(MouseEvent me)
   {
   }

   /**
    * Mouse exited.
    */
   public void mouseExited(MouseEvent me)
   {
   }

   /**
    * Mouse pressed.
    */
   public void mousePressed(MouseEvent me)
   {  showPopUp(me);
   }

   /**
    * Mouse released.
    */
   public void mouseReleased(MouseEvent me)
   {  showPopUp(me);
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   private void doNew()
   {  setWorld(
         new World(world.getWorldWidth(), world.getWorldHeight(), world.getScaleFactor()));
      menuBar.toggleGridLinesVisible(false);
      menuBar.toggleSmellVisible(false);
      menuBar.toggleLightVisible(false);
   }

   private void doOpen()
   {  File fileChosen = null;
      String fileName = null;
      World newWorld = null;
      try
      {  JFileChooser chooser = new JFileChooser();
         ExtensionFileFilter filter = new ExtensionFileFilter("world",
            "Agent Demonstrator Worlds");
         chooser.addChoosableFileFilter(filter);
         int usersAction = chooser.showOpenDialog(frame);
         if (usersAction == JFileChooser.APPROVE_OPTION)
         {  fileChosen = chooser.getSelectedFile();
            fileName = fileChosen.getName();
            ObjectInputStream in = 
               new ObjectInputStream(new FileInputStream(fileChosen));
            newWorld = (World) in.readObject();
            in.close();
            setWorld(newWorld);
            menuBar.toggleGridLinesVisible(newWorld.gridLinesAreVisible());
            menuBar.toggleSmellVisible(newWorld.smellIsVisible());
            menuBar.toggleLightVisible(newWorld.lightIsVisible());
         }
      }
      catch (ClassNotFoundException cnfe)
      {  JOptionPane.showMessageDialog(frame, "The file\n" + fileName +
            "\ncould not be opened!", "File open failure", JOptionPane.ERROR_MESSAGE);
      }
      catch (IOException ioe)
      {  JOptionPane.showMessageDialog(frame, "The file\n" + fileName +
            "\ncould not be opened!", "File open failure", JOptionPane.ERROR_MESSAGE);
      }
   }

   private void doSave()
   {  File fileChosen = null;
      String fileName = null;
      try
      {  JFileChooser chooser = new JFileChooser();
         ExtensionFileFilter filter = new ExtensionFileFilter("world",
            "Agent Demonstrator Worlds");
         chooser.addChoosableFileFilter(filter);
         boolean okToWrite = false;
         int usersAction = chooser.showSaveDialog(world);
         if (usersAction == JFileChooser.APPROVE_OPTION)
         {  fileName = filter.addExtension(chooser.getSelectedFile().getName());
            fileChosen = new File(fileName);
            if (fileChosen.exists())
            {  int overwrite = JOptionPane.showConfirmDialog(frame,
                  "The file\n" + fileName + "\nalready exists!\n" +
                  "Do you want to overwrite it?", "File exists",
                  JOptionPane.YES_NO_OPTION);
               okToWrite = (overwrite == JOptionPane.YES_OPTION);
            }
            else
            {  okToWrite = true;
            }
            if (okToWrite)
            {  ObjectOutputStream out = 
                  new ObjectOutputStream(new FileOutputStream(fileChosen));
               out.writeObject(world);
               out.close();
            }
         }
      }
      catch (IOException ioe)
      {  JOptionPane.showMessageDialog(frame, "The file\n" + fileName +
            "\ncould not be saved!", "File save failure", JOptionPane.ERROR_MESSAGE);
      }
   }

   private void doExit()
   {  int exitReply = JOptionPane.showConfirmDialog(
         frame, "Are you sure you want to exit?", "Exit?",
         JOptionPane.YES_NO_OPTION);
      if (exitReply == JOptionPane.YES_OPTION)
      {  System.exit(0);
      }
   }

   private void doGridLines()
   {  world.setGridLinesVisible(! world.gridLinesAreVisible());
      world.repaint();
   }

   private void doSmell()
   {  world.setSmellVisible(! world.smellIsVisible());
      world.repaint();
   }

   private void doLight()
   {  world.setLightVisible(! world.lightIsVisible());
      world.repaint();
   }

   private void doAbout()
   {  JOptionPane.showMessageDialog(frame, "Agent Demonstrator\n" +
         "Written for CS4071 Artificial Intelligence\n" +
         "By Derek Bridge, d.bridge@cs.ucc.ie", "About",
         JOptionPane.PLAIN_MESSAGE);
   }

   private void showPopUp(MouseEvent me)
   {  int evtX = me.getX();
      int evtY = me.getY();
      int x = evtX / world.getScaleFactor();
      int y = evtY / world.getScaleFactor();
      if (me.isPopupTrigger())
      {   new ObjectCreationMenu(frame, world, x, y).
             getPopUpMenu().show(world, evtX, evtY);
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private JFrame frame;
   private World world;
   private WorldMenuBar menuBar;
   private int time;
   private javax.swing.Timer clock;
   private PhysObject newObj;
   private SpeedDialog speedDialog;
   private int interval;

   private static final int DEFAULT_INTERVAL = 10;
}
