import java.awt.*; 
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

/** 
 * A class that represents pie chart objects (as JPanels) where segments
 * can be selected from a list and will highlight themselves.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class PieSelector
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new pie chart with a list whose clicked items
    * will be highlighted in the pie chart.
    *
    * @param theSegNames descriptions of each percentage.
    * @param thePercentages an array of percentages to be presented
    * as a pie chart; it is assumed that client code has ensured that
    * the sum of the values in the array is 100, and that no value
    * in the array is too small to show up.
    * @param the desired colours for each pie chart segment.
    *
    * It is assumed that client code has ensured that the three arrays
    * are the same length.
    */
   public PieSelector(String[] theSegNames,
      double[] thePercentages, Color[] theColours)
   {  super();

      /* Create a split pane with the list and the pie chart.
       */
      pieChart = new SelectablePieChart(thePercentages, theColours, -1);
      list = new JList(theSegNames);
      JScrollPane scrollPane = new JScrollPane(list);
      JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
         scrollPane, pieChart);
      add(splitPane);

      /* Register a listener with the list. List selection updates
         the pie chart.
      */
      list.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting()) // mouse wasn't yet at rest
            {  // do nothing
            }
            if (list.isSelectionEmpty()) 
            {  pieChart.setRaisedSegment(-1); // no segment is to be raised
            }
            else
            {  pieChart.setRaisedSegment(list.getSelectedIndex());
            }
         }
      });
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * A JList that displays data about each segment in the pie chart.
    */
   private JList list;

   /**
    * The pie chart being displayed.
    */
   private SelectablePieChart pieChart;

/* =======================================================================
       Test-DRIVER
   =======================================================================
*/
   public static void main(String[] args)
   {  JFrame frame = new JFrame("Winter Wombat Deaths");
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      Container contentPane = frame.getContentPane();
      String[] months = {"Dec", "Jan", "Feb", "Mar"};
      double[] winterWombatDeaths = {10, 20, 30, 40};
      Color[] cols = {Color.red, Color.green, Color.cyan, Color.yellow};
      PieSelector ps = new PieSelector(months, winterWombatDeaths, cols);
      contentPane.add(ps);
      frame.pack();
      frame.setVisible(true);
   }
}

