import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A class that represents one particular GUI for Till objects.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class TillGUI 
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new till GUI object.
    *
    * @param theController the controller (listener) for this gui.
    */
   public TillGUI(TillController theController)
   {  super();

      /* Store the till controller.
       */
      controller = theController;

      /* Create the four main parts of the GUI.
       */
      JPanel custP = createCustPanel();
      JPanel numPadP = createNumPadPanel();
      JPanel actionP = createActionPanel();
      JPanel tillTotP = createTillTotPanel();

      /* Layout the 4 parts.
       */
      setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      setLayout(gbl);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.insets = new Insets(20, 0, 0, 0);
      addComp(custP, this, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(numPadP, this, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(actionP, this, gbl, gbc, 2, 0, 1, 1, 0, 0);
      addComp(tillTotP, this, gbl, gbc, 3, 0, 1, 1, 0, 0);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Public constants-------------------------------------------------- */

   /**
    * The text that will appear on the numeric key pad Enter button.
    */
   public static final String ENTER = "Enter";

   /**
    * The text that will appear on the Submit button (for end of
    * customer transaction).
    */
   public static final String SUBMIT = "Submit";
  
   /**
    * The text that will appear on the Cancel button (to abort
    * customer transaction).
    */
   public static final String CANCEL = "Cancel";

/* --Getters----------------------------------------------------------- */

   /**
    * Return the value in the price textfield.
    *
    * @return the value in the price textfield (as an int).
    */
   public int getPriceEntered()
   {  return Integer.parseInt(priceTF.getText());
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Place a value into the price textfield.
    *
    * @param thePriceEntered the value to be placed into the price textfield
    * (an int).
    */
   public void setPriceEntered(int thePriceEntered)
   {  priceTF.setText("" + thePriceEntered);
   }

   /**
    * Place a value into the total owed textfield.
    *
    * @param theTotalOwed the value to be placed into the 
    * total owed textfield (an int).
    */
   public void setTotalOwed(int theTotalOwed)
   {  owedTF.setText("" + theTotalOwed);
   }

   /**
    * Place a value into the float textfield.
    *
    * @param theTillFloat the value to be placed into the float textfield
    * (an int).
    */
   public void setTillFloat(int theTillFloat)
   {  floatTF.setText("" + theTillFloat);
   }

   /**
    * Place a value into the till total textfield.
    *
    * @param theTillTotal the value to be placed into the 
    * till total textfield (an int).
    */
   public void setTillTotal(int theTillTotal)
   {  tillTotalTF.setText("" + theTillTotal);
   }

   /**
    * Place a value into the transaction count textfield.
    *
    * @param theTransactionCount the value to be placed into the 
    * transaction count textfield (an int).
    */
   public void setTransactionCount(int theTransactionCount)
   {  transTF.setText("" + theTransactionCount);
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Creates and lays out the components that deal with the current
    * customer: price of last item, and running total.
    */
   private JPanel createCustPanel()
   {  /* Create the fields and their labels.
       */
      priceLbl = new JLabel("Price:");
      priceTF = new JTextField(TF_WIDTH);
      priceTF.setEditable(false);
      priceTF.setBackground(Color.white);
      owedLbl = new JLabel("Total:");
      owedTF = new JTextField(TF_WIDTH);
      owedTF.setEditable(false);
      owedTF.setBackground(Color.white);

      /* Layout the fields and labels.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.WEST;
      gbc.insets = new Insets(4, 0, 0, 0);
      addComp(priceLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      addComp(priceTF, p, gbl, gbc, 0, 2, 1, 1, 0, 0);
      addComp(owedLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc,
         1, 1, 1, 1, 0, 0);
      addComp(owedTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
   
      return p;
   }

   /**
    * Creates and lays out the components that deal with the numeric
    * keypad.
    */
   private JPanel createNumPadPanel()
   {  /* Create the buttons of the numeric keypad.
       */
      digitB = new JButton[10];
      for (int i = 0; i < 10; i++)
      {  digitB[i] = new JButton("" + i);
         digitB[i].setBackground(Color.yellow);
      }
      enterB = new JButton(ENTER);
      enterB.setBackground(Color.cyan);

      /* Layout the buttons.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;
      addComp(digitB[0], p, gbl, gbc, 0, 0, 1, 3, 1, 1);
      for (int i = 1; i < 10; i++)
      {  addComp(digitB[i], p, gbl, gbc, (i + 2) / 3, 
            (i % 3) - 1, 1, 1, 1, 1);
      }
      addComp(enterB, p, gbl, gbc, 0, 3, 4, 1, 1, 1);

      /* Register a listener for the numeric buttons.
         They all share the same listener object, and it is defined
         as an anonymous class.
         Its handler simply updates the GUI's price field by appending
         the latest digit that the user has pressed onto the price so 
         far.
       */
      ActionListener lstnr = new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String btnText = ae.getActionCommand();
            int digit = Integer.parseInt(btnText);
            int priceSoFar = getPriceEntered();
            int newPrice = priceSoFar * 10 + digit;
            setPriceEntered(newPrice);
         }
      };
      for (int i = 0; i < 10; i++)
      {  digitB[i].addActionListener(lstnr);
      }

      /* Register a listener for the Enter button.
       */
      enterB.addActionListener(controller);

      return p;
   }

   /**
    * Creates and lays out the components that deal with till actions:
    * submit and cancel.
    */
   private JPanel createActionPanel()
   {  /* Create the buttons that deal with the main till actions.
       */
      submitB = new JButton(SUBMIT);
      submitB.setBackground(Color.green);
      cancelB = new JButton(CANCEL);
      cancelB.setBackground(Color.red);

      /* Layout the buttons.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      p.setLayout(new BoxLayout(p, BoxLayout.X_AXIS));
      p.add(submitB);
      p.add(Box.createHorizontalStrut(4));
      p.add(cancelB);

      /* Register listeners for the Submit and Cancel buttons.
       */
      submitB.addActionListener(controller);
      cancelB.addActionListener(controller);

      return p;
   }

   /**
    * Creates and lays out the components that deal with the till totals:
    * initial float, total takings and number of transactions.
    */
   private JPanel createTillTotPanel()
   {  /* Create the textfields and their labels for showing the till
         information.
       */
      floatLbl = new JLabel("Float:");
      floatTF = new JTextField(TF_WIDTH);
      floatTF.setEditable(false);
      floatTF.setBackground(Color.white);
      tillTotalLbl = new JLabel("Till Total:");
      tillTotalTF = new JTextField(TF_WIDTH);
      tillTotalTF.setEditable(false);
      tillTotalTF.setBackground(Color.white);
      transLbl = new JLabel("Transactions:");
      transTF = new JTextField(TF_WIDTH);
      transTF.setEditable(false);
      transTF.setBackground(Color.white);

      /* Layout the textfield and their labels.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.WEST;
      gbc.insets = new Insets(4, 0, 0, 0);
      addComp(floatLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      addComp(floatTF, p, gbl, gbc, 0, 2, 1, 1, 0, 0);
      addComp(tillTotalLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc,
         1, 1, 1, 1, 0, 0);
      addComp(tillTotalTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
      addComp(transLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      addComp(transTF, p, gbl, gbc, 2, 2, 1, 1, 0, 0);
   
      return p;
   }

   /**
    * Utility method for adding components to a container using
    * grid bag layout constraints.
    *
    * @param comp the component being added.
    * @param cont the container we're adding to.
    * @param gbl the layout manager.
    * @param gbc the layout manager's constraints object.
    * @param row the row we're positioning in.
    * @param col the column we're positioning in.
    * @param numRows the number of rows this component is to span.
    * @param numCols the number of columns this component is to span.
    * @param weightx horizontal resize weight.
    * @param weighty vertical resize weight.
    */
   private void addComp(Component comp, Container cont, GridBagLayout gbl,
      GridBagConstraints gbc, int row, int col, int numRows, int numCols,
      double weightx, double weighty)
   {  gbc.gridx = col;
      gbc.gridy = row;
      gbc.gridwidth = numCols;
      gbc.gridheight = numRows;
      gbc.weightx = weightx;
      gbc.weighty = weighty;

      cont.add(comp, gbc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The till controller (i.e. a listener for use with the
    * non-numeric buttons).
    */
   private TillController controller;

   /**
    * The label that sits alongside the price textfield.
    */
   private JLabel priceLbl;

   /**
    * The textfield in which is displayed the price of the item that
    * is currently being rung up.
    */
   private JTextField priceTF;

   /**
    * The label that sits alongside the owed textfield.
    */
   private JLabel owedLbl;

   /**
    * The textfield in which is displayed the amount owed by the
    * current customer.
    */
   private JTextField owedTF;

   /**
    * The array of buttons labelled by digits.
    */
   private JButton[] digitB;

   /**
    * The Enter button for use with the numeric buttons.
    */
   private JButton enterB;

   /**
    * The button for indicating that all of this customer's
    * items have been rung up.
    */
   private JButton submitB;

   /**
    * The button for aborting the current customer.
    */
   private JButton cancelB;

   /**
    * The label that sits alongside the float textfield.
    */
   private JLabel floatLbl;

   /**
    * The textfield in which is displayed the amount of float in
    * the till.
    */
   private JTextField floatTF;

   /**
    * The label that sits alongside the till total textfield.
    */
   private JLabel tillTotalLbl;

   /**
    * The textfield in which is displayed the total cash in the
    * till.
    */
   private JTextField tillTotalTF;

   /**
    * The label that sits alongside the transaction count textfield.
    */
   private JLabel transLbl;

   /**
    * The textfield in which is displayed a count of the transactions
    * (customers) processed by this till.
    */
   private JTextField transTF;

   /**
    * The width of the textfields (in columns).
    */
   private static final int TF_WIDTH = 10;
}


