/**
 * A class that represents simple time objects.
 *
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class Time
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new time object for the specified hour,
    * minutes and seconds.
    *
    * @param theHour this time's hour (24-hour clock)
    * @param theMins this time's minutes past the hour
    * @param theSecs this time's seconds past the minute
    */
   public Time(int theHour, int theMins, int theSecs)
   {  hours = theHour;
      mins = theMins;
      secs = theSecs;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Public class constants-------------------------------------------- */

   /**
    * The number of hours per day.
    */
   public static final int HOURS_PER_DAY = 24;

   /**
    * The number of seconds per minute.
    */
   public static final int SECS_PER_MIN = 60;

   /**
    * The number of minutes per hour.
    */
   public static final int MINS_PER_HOUR = 60;

   /**
    * The number of seconds per hour.
    */
   public static final int SECS_PER_HOUR = SECS_PER_MIN * MINS_PER_HOUR;

/* --Getters----------------------------------------------------------- */

   /**
    * Returns true if this time is equal to the specified time; 
    * false otherwise.
    *
    * @param aTime the time being compared with this time.
    * @return true if and only if this time equals the specified 
    * time.
    */
   public boolean isEqualsTo(Time aTime)
   {  return this.hours == aTime.hours && 
         this.mins == aTime.mins && this.secs == aTime.secs;
   }

   /**
    * Returns true if this time is before the specified time;
    * false otherwise.
    *
    * @param aTime the time being compared with this time.
    * @return true if and only if this time is strictly earlier
    * than the specified time.
    */
   public boolean isBefore(Time aTime)
   {  return ((this.hours < aTime.hours) ||
              (this.hours == aTime.hours && this.mins < aTime.mins) ||
              (this.hours == aTime.hours && this.mins == aTime.mins &&
               this.secs < aTime.secs));
   }

   /**
    * Returns the number of seconds there are between this time
    * and the specified time. 
    *
    * @param aTime the end time of the interval that is being
    * computed. (It is assumed that aTime is after this time.)
    * @return the number of seconds there are between this time
    * and the specified time.
    */
   public int getIntervalInSecsUntil(Time aTime)
   {  return 
         (aTime.hours * SECS_PER_HOUR + 
          aTime.mins * SECS_PER_MIN + aTime.secs) - 
         (this.hours * SECS_PER_HOUR +
          this.mins * SECS_PER_MIN + this.secs);
   }

   /**
    * Returns a string that expresses this time as a `clock time',
    * i.e. hours, minutes and seconds.
    *
    * @return this time as a `clock time'.
    */
   public String getClockTime()
   {  return hours + ":" + mins + ":" + secs;
   }

   /**
    * Returns a string that expresses this time as a `military time',
    * i.e. hours (in 12-hour clock, not 24-hour clock), minutes and
    * a.m. or p.m.
    *
    * @return this time as a `military time'.
    */
   public String getMilitaryTime()
   {  String milTime;
      String partOfDay = getPartOfDay();
      if (partOfDay.equals(MIDNIGHT_TEXT) || partOfDay.equals(MIDDAY_TEXT))
      {  milTime = partOfDay;
      }
      else
      {  int milTimeHour = hours;
         if (partOfDay.equals(AFTERNOON_TEXT))
         {  milTimeHour -= MIDDAY_IN_HOURS;
         }
         milTime = milTimeHour + ":" + mins + ":" + secs + 
            " " + partOfDay;
      }
      return milTime;
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Returns the number of hours that comprises this time.
    *
    * @return the hours for this time.
    */
   private int getHours()
   {  return hours;
   }

   /**
    * Returns the number of minutes past the hour that comprise
    * this time.
    *
    * @return the minutes past the hour for this time.
    */
   private int getMins()
   {  return mins;
   }

   /**
    * Returns the number of seconds past the minute that comprise
    * this time.
    *
    * @return the seconds past the minute for this time.
    */
   private int getSecs()
   {  return secs;
   }

   /**
    * Returns a string that indicates this time's part of day,
    * i.e. midnight, morning, midday or afternoon.
    *
    * @return this time's part of day.
    */
   private String getPartOfDay()
   {  String partOfDay;
      int timeInSecs =  hours * SECS_PER_HOUR +
         mins * SECS_PER_MIN + secs;
      if (timeInSecs == MIDNIGHT_IN_SECS)
      {  partOfDay = MIDNIGHT_TEXT;
      }
      else if (timeInSecs < MIDDAY_IN_SECS)
      {  partOfDay = MORNING_TEXT;
      }
      else if (timeInSecs == MIDDAY_IN_SECS)
      {  partOfDay = MIDDAY_TEXT;
      }
      else // timeInSecs > MIDDAY_IN_SECS
      {  partOfDay = AFTERNOON_TEXT;
      }
      return partOfDay;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   /**
    * The hours.
    */
   private int hours;

   /**
    * The minutes.
    */
   private int mins;

   /**
    * The seconds.
    */
   private int secs;

   /**
    * The word used in military time for midnight.
    */
   private static final String MIDNIGHT_TEXT = "midnight";

   /**
    * The word used in military time for morning.
    */
   private static final String MORNING_TEXT = "a.m.";

   /**
    * The word used in military time for midday.
    */
   private static final String MIDDAY_TEXT = "midday";
   
   /**
    * The word used in military time for afternoon.
    */
   private static final String AFTERNOON_TEXT = "p.m.";

   /**
    * Midday expressed as hours since midnight.
    */
   private static final int MIDDAY_IN_HOURS = HOURS_PER_DAY / 2;

   /**
    * Midnight expressed as seconds since midnight.
    */
   private static final int MIDNIGHT_IN_SECS = 0;

   /**
    * Midday expressed as seconds since midnight.
    */
   private static final int MIDDAY_IN_SECS = MIDDAY_IN_HOURS * SECS_PER_HOUR;
}



   