import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A class that illustrates a GUI with an updateable JList.
 * This is still a simple case: we use the default list model:
 * insertions and deletions affect the model, which informs the
 * GUI
 * @author Derek Bridge
 */

public class UpdateableList
   extends JFrame
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new frame with a list displayed on it.
    */
   public UpdateableList()
   {  super("Demo of simple updateable list");

      /* We create a list model to hold the data.
         We use the class definition that Swing supplies for
         this purpose: DefaultListModel.
       */
      model = new DefaultListModel();
      model.addElement("item0");
      model.addElement("item1");
      model.addElement("item2");
      model.addElement("item3");
      model.addElement("item4");
      model.addElement("item5");
      model.addElement("item6");
      model.addElement("item7");
      itemCount = 8;

      /* Now we create the JList using the model.
      */
      list = new JList(model);
      list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      list.setVisibleRowCount(4);
      list.setSelectedIndex(0);

      /* We register a listener that says what to do when the user
         selects an item in the list.
      */
      list.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting())
            {  // do nothing
            }
            else 
            {  int index = list.getSelectedIndex();
               itemLbl.setText((String) model.elementAt(index));
            }
         }
      });

      /* We create two buttons, with their own listeners.
         When these buttons are clicked, their listeners update the
         model. Due to Swing's MVC architecture, changes to the model
         are then reflected in the view.
      */
      insBtn = new JButton("Insert");
      insBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  /* We'll insert into the list at a point that comes just 
               after whatever is currently selected.
            */
            int index = list.getSelectedIndex();
            int size = model.getSize();
            /* If user made no selection (i.e. index == -1, e.g. list 
               is empty) or if last position is selected, add to end of 
               list.
            */
            if (index == -1 || index + 1 == size)
            {  // Update the model; view will also be updated
               model.addElement("item" + itemCount++); 
               // Update the JList; will foward to model; view will be updated
               list.setSelectedIndex(size); 
            }
            /* Otherwise insert after currently selected.
            */
            else
            {  model.insertElementAt("item" + itemCount++, index + 1);
               list.setSelectedIndex(index + 1);
            }
            delBtn.setEnabled(true);
         }
      });

      delBtn = new JButton("Delete");
      delBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = list.getSelectedIndex();
            // Update model; view will also be updated
            model.remove(index);          
            if (model.getSize() == 0) // nothing's left
            {  delBtn.setEnabled(false);
            }
            else
            {  // Highlight a new item
               if (index == model.getSize()) // removed item was the last
               {  index--;
               }
               list.setSelectedIndex(index);
            }
         }
      });

      /* Finally, add everything to the content pane.
      */
      JScrollPane scrollPane = new JScrollPane(list);
      itemLbl = new JLabel("item0");

      Container contentPane = getContentPane();
      contentPane.setLayout(new FlowLayout());
      contentPane.add(scrollPane);
      contentPane.add(itemLbl);
      contentPane.add(insBtn);
      contentPane.add(delBtn);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   private JList list;
   private JLabel itemLbl;
   private JButton insBtn;
   private JButton delBtn;
   private DefaultListModel model;

   private int itemCount;

/* =======================================================================
       Test DRIVER
   =======================================================================
*/

   public static void main(String[] args)
   {  JFrame frame = new UpdateableList();
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      frame.setBounds(100, 100, 600, 200);
      frame.setVisible(true);
   }
}
