import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A class that illustrates a GUI with an updateable JList.
 * We use the default list model so that the GUI is informed
 * of insertions and deletions on the model. But we also
 * listen out for these model insertions/deletions and do
 * some additional processing of our own when they arise.
 * @author Derek Bridge
 */

public class CustomisedList
   extends JFrame
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new frame with a list displayed on it.
    */
   public CustomisedList()
   {  super("Demo of customised updateable list");

      /* We still use a DefaultListModel and most of the code
         is the same.
         Indeed, none of the code has changed from here....
       */
      model = new DefaultListModel();
      model.addElement("item0");
      model.addElement("item1");
      model.addElement("item2");
      model.addElement("item3");
      model.addElement("item4");
      model.addElement("item5");
      model.addElement("item6");
      model.addElement("item7");
      itemCount = 8;
      list = new JList(model);
      list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      list.setVisibleRowCount(4);
      list.setSelectedIndex(0);
      list.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  if (lse.getValueIsAdjusting())
            {  // do nothing
            }
            else 
            {  int index = list.getSelectedIndex();
               itemLbl.setText((String) model.elementAt(index));
            }
         }
      });
      insBtn = new JButton("Insert");
      insBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = list.getSelectedIndex();
            int size = model.getSize();
            if (index == -1 || index + 1 == size)
            {  model.addElement("item" + itemCount++); 
               list.setSelectedIndex(size); 
            }
            else
            {  model.insertElementAt("item" + itemCount++, index + 1);
               list.setSelectedIndex(index + 1);
            }
            delBtn.setEnabled(true);
         }
      });
      delBtn = new JButton("Delete");
      delBtn.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = list.getSelectedIndex();
            model.remove(index);          
            if (model.getSize() == 0) 
            {  delBtn.setEnabled(false);
            }
            else
            {  if (index == model.getSize()) 
               {  index--;
               }
               list.setSelectedIndex(index);
            }
         }
      });

      /* ...to here!

         Now we register a listener. This is registered WITH THE MODEL.
         It must implement the interface ListDataListener. It must
         include methods contentsChanged, intervalAdded and intervalRemoved.
         When the list model is updated, this creates a ListDataEvent.
         And one of the methods in the listener will be executed.
      */
      model.addListDataListener(new ListDataListener()
      {  public void contentsChanged(ListDataEvent lde)
         {  // do nothing
         }

         public void intervalAdded(ListDataEvent lde)
         {  sizeLbl.setText(model.getSize() + " items");
         }

         public void intervalRemoved(ListDataEvent lde)
         {  sizeLbl.setText(model.getSize() + " items");
         }
      });

      /* Finally, add everything to the content pane.
      */
      JScrollPane scrollPane = new JScrollPane(list);
      itemLbl = new JLabel("item0");
      sizeLbl = new JLabel(model.getSize() + " items");

      Container contentPane = getContentPane();
      contentPane.setLayout(new FlowLayout());
      contentPane.add(scrollPane);
      contentPane.add(itemLbl);
      contentPane.add(insBtn);
      contentPane.add(delBtn);
      contentPane.add(sizeLbl);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   private JList list;
   private JLabel itemLbl;
   private JButton insBtn;
   private JButton delBtn;
   private JLabel sizeLbl;
   private DefaultListModel model;

   private int itemCount;

/* =======================================================================
       Test DRIVER
   =======================================================================
*/

   public static void main(String[] args)
   {  JFrame frame = new CustomisedList();
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      frame.setBounds(100, 100, 600, 200);
      frame.setVisible(true);
   }
}
