import java.util.Vector;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A class that illustrates a GUI with a JList.
 * This is the simplest case: once the list has been created,
 * the user can select items and this causes an event handler
 * to be run. But crucially we do not attempt to update the
 * contents of the list (no insertions, no deletions, no
 * modifications).
 * @author Derek Bridge
 */

public class SimplestList
   extends JFrame
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new frame with a list displayed on it.
    */
   public SimplestList()
   {  super("Demo of simple unchangeable list");

      /* We'll create a Vector containing some Strings.
         We could alternatively use an array of Objects.
       */
      v = new Vector();
      v.addElement("item0");
      v.addElement("item1");
      v.addElement("item2");
      v.addElement("item3");
      v.addElement("item4");
      v.addElement("item5");
      v.addElement("item6");
      v.addElement("item7");

      /* Now we create the JList from the vector (or array of Objects).
         We specify a selection mode, which can be one of 
         SINGLE_SELECTION, SINGLE_INTERVAL_SELECTION or 
         MULTIPLE_INTERVAL_SELECTION.
         We specify how many items will be visible.
         And we specify which item will be selected when the list is
         first displayed.
      */
      list = new JList(v);
      list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      list.setVisibleRowCount(4);
      list.setSelectedIndex(0);

      /* We register a listener that says what to do when the user
         selects an item in the list (ListSelectionEvent).
         For simplicity, we use an anonymous class for this.
         It must implement ListSelectionListener, which means it must
         include a method called valueChanged.
      */
      list.addListSelectionListener(new ListSelectionListener()
      {  public void valueChanged(ListSelectionEvent lse)
         {  /* Conventionally, the handler first checks that the event
               is worth responding to. 
               If this test comes out true, the event was created while
               the mouse was being dragged. We should ignore this event.
               We really only need to deal with the case where the mouse
               has finally come to rest.
            */
            if (lse.getValueIsAdjusting())
            {  // do nothing
            }
            else 
            {  /* We find out the selected index. (There's only one because
                  our selection mode is SINGLE_SELECTION; had we used the
                  other selection modes, we would need to get the selections
                  into an array.)
               */
               int index = list.getSelectedIndex();
               /* Then we do whatever we want to do with this item in the
                  Vector (remembering that, in this simple example, it's no 
                  good updating the Vector --- that's the subject of other
                  examples.)
               */
               itemLbl.setText((String) v.elementAt(index));
            }
         }
      });

      /* Finally, we put the list into a scroll pane.
         Then we place the scollable list into the contentPane of the
         frame, along with a JLabel.
      */
      JScrollPane scrollPane = new JScrollPane(list);
      itemLbl = new JLabel("item0");

      Container contentPane = getContentPane();
      contentPane.setLayout(new FlowLayout());
      contentPane.add(scrollPane);
      contentPane.add(itemLbl);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   private JList list;
   private JLabel itemLbl;
   private Vector v;

/* =======================================================================
       Test DRIVER
   =======================================================================
*/

   public static void main(String[] args)
   {  JFrame frame = new SimplestList();
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      frame.setBounds(100, 100, 600, 200);
      frame.setVisible(true);
   }
}
