import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;

/**
 * A class that represents burger bar GUIs, i.e. gui objects that allow
 * staff to take and process customer orders.
 * @author Derek Bridge
 */
public class BurgerBarGUI
   extends JFrame
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a window object containing components for ordering
    * dishes and receiving money into a till.
    *
    * @param theBurgerBar the burger bar whose gui we are creating.
    */
   public BurgerBarGUI(BurgerBar theBurgerBar)
   {  super("Derek's Burger Bar");
      /* Store this GUI's burger bar.
       */
      burgerBar = theBurgerBar;      

      /* Create the three main parts of the GUI
       */
      JPanel logoP = createLogoPanel();
      JPanel orderP = createOrderPanel();
      Till till = theBurgerBar.getTill();
      ITillGUI tillGUI = new BurgerBarTillGUI(till);
      till.setGUI(tillGUI);
      JPanel tillP = (JPanel) tillGUI;
      JPanel actionP = createActionPanel();

      /* Layout the 4 parts.
       */
      Container contentPane = getContentPane();
      contentPane.setBackground(BACK_COLOUR);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      contentPane.setLayout(gbl);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.insets = new Insets(20, 20, 20, 20);
      addComp(logoP, contentPane, gbl, gbc, 0, 0, 1, 2, 1, 1);
      addComp(orderP, contentPane, gbl, gbc, 1, 0, 1, 1, 1, 1);
      addComp(tillP, contentPane, gbl, gbc, 1, 1, 1, 1, 1, 1);
      addComp(actionP, contentPane, gbl, gbc, 2, 0, 1, 2, 1, 1);

      /* Listener for this window, so program terminates when
         window is closed.
       */
      addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Creates a panel that contains a logo.
    */
   public JPanel createLogoPanel()
   {  JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      JLabel logo = new JLabel(LOGO);
      Font font = LOGO_FONT;
      logo.setFont(font);
      logo.setForeground(FORE_COLOUR);
      p.add(logo); 
      return p;
   }

   /**
    * Creates a panel with buttons for choosing meals,
    * a list of chosen items and a delete button.
    */
   private JPanel createOrderPanel()
   {  /* Create the buttons for each item in the bill of fare.
         Lay them out by type, e.g. all burgers in one row.
       */
      JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;

      Vector bof = burgerBar.getBillOfFare();
      for (int i = 0; i < bof.size(); i++)
      {  Vector dishes = (Vector) bof.elementAt(i);
         for (int j = 0; j < dishes.size(); j++)
         {  AbstractDish dish = (AbstractDish) dishes.elementAt(j);
            JButton b = new JButton(dish.getName());
            b.setBackground(BACK_COLOUR);
            b.setForeground(FORE_COLOUR);
            addComp(b, p, gbl, gbc, i, j, 1, 1, 1, 1);
            b.addActionListener(new AddItemToOrderListener(dish, burgerBar));
         }
      }
      /* Create the list for displaying the customer's order
         and the button for deleting parts of that order.
       */
      gbc.fill = GridBagConstraints.NONE;
      gbc.insets = new Insets(10, 10, 10, 10);
      list = new JList(burgerBar.getCurrentOrder());
      list.setForeground(FORE_COLOUR);
      list.setSelectionBackground(FORE_COLOUR);
      JScrollPane sp = new JScrollPane(list);
      list.setVisibleRowCount(5); 
      list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      addComp(sp, p, gbl, gbc, GridBagConstraints.RELATIVE, 0,
         1, GridBagConstraints.REMAINDER, 1, 1);
      JButton deleteB = new JButton(DELETE_ITEM);
      deleteB.setBackground(BACK_COLOUR);
      deleteB.setForeground(FORE_COLOUR);
      addComp(deleteB, p, gbl, gbc, GridBagConstraints.RELATIVE, 0, 
         1, GridBagConstraints.REMAINDER, 1, 1); 

      deleteB.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = list.getSelectedIndex();
            if (index != -1)
            {  burgerBar.removeItemFromOrder(index);
            }
         }
      });

      return p;
   }

   /**
    * Creates a panel with a button that offers the action of
    * moving to the next customer.
    */
   private JPanel createActionPanel()
   {  JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      JButton nextB = new JButton(NEXT_CUST);
      nextB.setBackground(BACK_COLOUR);
      nextB.setForeground(FORE_COLOUR);
      p.add(nextB);

      nextB.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  burgerBar.concludeOrder();
         }
      });

      return p;
   }

   /**
    * Utility method for adding components to a container using
    * grid bag layout constraints.
    *
    * @param comp the component being added.
    * @param cont the container we're adding to.
    * @param gbl the layout manager.
    * @param gbc the layout manager's constraints object.
    * @param row the row we're positioning in.
    * @param col the column we're positioning in.
    * @param numRows the number of rows this component is to span.
    * @param numCols the number of columns this component is to span.
    * @param weightx horizontal resize weight.
    * @param weighty horizontal resize weight.
    */
   private void addComp(Component comp, Container cont, GridBagLayout gbl,
      GridBagConstraints gbc, int row, int col, int numRows, int numCols,
      double weightx, double weighty)
   {  gbc.gridx = col;
      gbc.gridy = row;
      gbc.gridwidth = numCols;
      gbc.gridheight = numRows;
      gbc.weightx = weightx;
      gbc.weighty = weighty;

      cont.add(comp, gbc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   private BurgerBar burgerBar;
   private JList list;

   private static final String NEXT_CUST = "Next Customer";
   private static final String DELETE_ITEM = "Delete Item";
   private static final String LOGO = "Derek's Burger Bar";
   private static final Font LOGO_FONT = new Font("Serif", Font.ITALIC, 36);

   private static final Color BACK_COLOUR = Color.white;
   private static final Color FORE_COLOUR = new Color(255, 0, 153);
}
