package consoleIO;

import utilities.*;

/**
 * A class that represents simple menu items.
 * ConsoleMenuItem objects have a description, which is a string that
 * can be displayed on the console when the menu is being displayed.
 * They also contain an object that will launch appropriate processing 
 * when the menu item is slected.
 * @author Derek Bridge
 */

public class ConsoleMenuItem
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new console menu item object with the specified 
    * description and the specified Command object that says
    * what processing to do when this item is chosen in a menu.
    *
    * @param theDesc the menu description of this item; must be non-null
    * and non-empty.
    * @param theCommand the command object that specifies the processing
    * that corresponds to this menu item. Command is an abstract
    * class, so the parameter object will actually be an instance of
    * a concrete subclass of Command. Must be non-null.
    */
   public ConsoleMenuItem(final String theDesc, final Command theCommand)
   {  if (theDesc == null || theDesc.trim().equals(""))
      {  throw new PreconditionException(
            "ConsoleMenuItem Constructor Precondition Error: invalid menu " +
            "item description: " + theDesc);
      }
      if (theCommand == null)
      {  throw new PreconditionException(
            "ConsoleMenuItem Constructor Precondition Error: invalid menu " +
            "item command object: " + theCommand);
      }

      desc = theDesc;
      command = theCommand;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Returns the textual description of this menu item, typically
    * so that it can be displayed in a menu.
    *
    * @return the item's textual description.
    */
   public String getDesc()
   {  return desc;
   }      

   /**
    * When this menu item gets selected within a menu, certain processing
    * should happen. Each menu item contains a Command object, whose
    * execute() method specifies this processing. The invoke()
    * method is what launches the command object's execute() method.
    */
   public void invoke()
   {  command.execute();
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   
  
   // The textual description of this menu item.
   private String desc; 

   // The command object whose execute() method specifies what 
   // processing to do when this menu item is selected.
   private Command command; 
}
