package web;

import consoleIO.*;
import utilities.*;
import datastructs.*;
import java.net.*;
import java.io.*;
import java.util.Enumeration;

/**
 * Objects of this class are menus of URLs.  Users select URLs
 * from these menus and the source of the corresponding file is
 * downloaded and directed to a specified print stream.
 * These menus use console input/output.
 * @author Derek Bridge
 */

public class URLMenu
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new URL menu object that uses console input/output.
    *
    * @param aURLSeq a sequence of URL objects; these will be the
    * menu items; must be non-null; each item must be a URL.
    * @param theURLFile the file from whence these URLs came; must
    * be non-null, must exist and must be a file not a directory.
    * @exception NotAURLFileException if the file that is read in to
    * form the menu contents doesn't contain only URLs.
    */
   public URLMenu(final Sequence aURLSeq, final File theURLFile)
      throws NotAURLFileException
   {  if (aURLSeq == null)
      {  throw new PreconditionException(
            "URLMenu Constructor Precondition Error: invalid sequence: " +
            aURLSeq);
      }
      if (theURLFile == null || ! theURLFile.exists() || 
         theURLFile.isDirectory())
      {  throw  new PreconditionException(
            "URLMenu Constructor Precondition Error: invalid file: " +
            theURLFile);
      }

      urlSeq = aURLSeq;
      urlFile = theURLFile;
      try
      {  /* Create the menu.
          */
         menu = new ConsoleMenu(TITLE, PROMPT);
         /* Now create the menu items. Menu maintenance options first.
          */
         menu.addItem(new ConsoleMenuItem(INSERT_TEXT,
            new InsertCommand()));
         menu.addItem(new ConsoleMenuItem(SAVE_AND_EXIT_TEXT,
            new SaveExitCommand()));
         /* The other items consist of URL objects in the give sequence.
          */
         Enumeration enum = aURLSeq.elements();
         while (enum.hasMoreElements())
         {  URL nextURL = (URL) enum.nextElement();
            menu.addItem(new ConsoleMenuItem(nextURL.toString(),
               new SelectURLCommand(nextURL)));
         }
      }
      catch (ClassCastException e)
      {  throw new NotAURLFileException();
      }
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* ------Getters------------------------------------------------------- */

   /**
    * Displays the menu on the console.
    */
   public void show()
   {  menu.show();
   }

   /**
    * Get the user to choose a URL or other action from the menu. 
    * The Command object associated with the choice will then
    * determine the subsequent processing.
    */
   public void invokeChosenItem() 
   {  menu.invokeChosenItem();
   }

/* =======================================================================
       MEMBER CLASSES
   =======================================================================
*/  

   /** 
    * The class of command objects associated with URLs in a URL
    * menu.
    * When its execute() method is run, a connection is made and the 
    * file that corresponds to this URL has its source downloaded and 
    * directed to a print stream.
    */
   private class SelectURLCommand
      extends Command
   {
      /*
       * Allocates this kind of Command object.
       *
       * @param theURL the URL that this command object refers to;
       * must be non-null and a URL; this precondition is enforced
       * by the client code.
       */
      private SelectURLCommand(final URL theURL)
      {  url = theURL;
      }

      /*
       * The correct response to a user choosing a URL from a URL menu
       * is to create an object that makes a connction and downloads
       * and directs the corresponding web file to a print stream.
       * The menu is then redisplayed.
       */
      public void execute()
      {  try
         {  /* Download the content of the selected URL.
             */
            URLDownLoader dl = new URLDownLoader(url);
            dl.download(CHOSEN_PRINT_STREAM);
         }
         catch (DownLoadException e)
         {  System.out.println("The connection to your file has not succeeded.");
            System.out.println("This might be because the URL is not valid,");
            System.out.println("the file no longer exists, the network is");
            System.out.println("down, your machine cannot issue Interner");
            System.out.println("requests, the machine on which the file");
            System.out.println("resides isn't responding, among other");
            System.out.println("reasons. It may be worth trying again later.");
         }
         /* Redisplay the menu and take another user choice.
          */
         show();
         invokeChosenItem();
      }

      private URL url;
   }

   /** 
    * The class of command objects associated with the `insert URL' action
    * in a URL menu.
    * When its execute() method is run, the user is prompted for  a new
    * URL and this is added to the menu.
    */
   private class InsertCommand
      extends Command
   {
      /*
       * Allocates this kind of Command object.
       */
      private InsertCommand()
      { 
      }

      /*
       * The correct response to a user choosing `insert URL' from a URL menu
       * is to prompt for a new URL from the user and add it to the menu.
       * The menu is then redisplayed.
       */
      public void execute() 
      {  try
         {  /* Prompt for and accept a new URL from the user.
               If a URL object cannot be constructed from the user's
               input, an exception is thrown.
             */
            System.out.print(NEW_URL_PROMPT);
            String usersSpec = new ConsoleInput().readLine(); 
            URL usersURL = new URL(usersSpec);
            urlSeq.addLast(usersURL);
            menu.addItem(new ConsoleMenuItem(usersSpec,
               new SelectURLCommand(usersURL)));
         }
         catch (MalformedURLException e)
         {  System.out.println("You have not entered a legal URL.");
            System.out.println("Check each part.");
         }
         catch (IOException e)
         {  System.out.println("There was a problem in reading your input.");
            System.out.println("Check your system configuration.");
         }
         /* The menu is redisplayed and the user makes another choice.
          */
         show();
         invokeChosenItem();
      }
   }

   /** 
    * The class of command objects associated with the `save and exit' action
    * in a URL menu.
    * When its execute() method is run, each URL on the menu is written
    * to the file from which they originally came and the program is exited.
    */
   private class SaveExitCommand
      extends Command
   {  
      /*
       * Allocates this kind of Command object.
       */
      private SaveExitCommand()
      {
      }

      /*
       * The correct response to a user choosing `save/exit' from a URL menu
       * is to write the URLs in the menu to the file from which they came
       * and exit the program.
       */
      public void execute()
      {  try
         {  URLFile file = new URLFile(urlFile);
            file.writeFromSeq(urlSeq);
            System.out.println("URLs have been saved. Exiting program.");
            System.exit(0);
         }
         catch (NotAURLFileException e)
         {  /* Should probably fail more gracefully, e.g. save the urls to
               a temporary file, if possible. Didn't get round to writing
               the code to do that.
             */
            System.out.println("Sorry. Couldn't write your URLs because");
            System.out.println("they weren't all URLs. Tough. Exiting anyway.");
            System.exit(1);
         }
         catch (URLFileReadWriteException e)
         {  /* Should probably fail more gracefully, e.g. save the urls to
               a temporary file, if possible. Didn't get round to writing
               the code to do that.
             */
            System.out.println("Sorry. Couldn't write your URLs because");
            System.out.println("had a file problem. Tough. Exiting anyway.");
            System.exit(1);
         }
      }

   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   
  
   private ConsoleMenu menu; // the menu of URL manipulation options.
   private Sequence urlSeq; // the sequence of URLs that appears on the menu.
   private File urlFile; // the file from which these URLs came/will be saved.

   private static final String TITLE = "URL Menu";
   private static final String PROMPT = "Enter a number from the menu: ";
   private static final String INSERT_TEXT = "Add a new URL.";
   private static final String SAVE_AND_EXIT_TEXT =
      "Save these URLs and exit.";
   private static final String NEW_URL_PROMPT = 
      "Enter the URL you wish to insert: ";
   private static final PrintStream CHOSEN_PRINT_STREAM = System.out;
}
