package web;

import java.io.File;
import consoleIO.*;
import datastructs.*;
import utilities.*;

/**
 * A class that represents low quality web browsers.
 * @author Derek Bridge
 */
public class CheapBrowser
{
   /**
    * Allocates a new cheap and nasty browser object.
    */
   public CheapBrowser()
   {
   }

   /**
    * The user picks a file from the start directory. This should
    * be a file of URLs. This file is read in and its contents form
    * the basis of a menu. So the user can then pick a URL whose
    * source s/he would like to have downloaded for viewing.
    *
    * @param startDir the directory from which the user will select
    * a file of URLs; must be non-null and a directory, not a file.
    */
   public void launch(File theStartDir)
   {  if (theStartDir == null || ! theStartDir.isDirectory())
      {  throw new PreconditionException(
            "CheapBrowser.launch Precondition Error: invalid directory: " +
            theStartDir);
      }
      
      File currentDir = theStartDir;

      boolean isFileReadIn = false;
      do
      {  try
         {  /* The user must select a file of URLs. We create, show and
               obtain a user selection using a file dialog object.
            */
            ConsoleFileDialog cfd = new ConsoleFileDialog(
               "Select a URL file", ConsoleFileDialog.LOAD, currentDir);
            cfd.show();
            File chosenFile = cfd.getUsersChoice(); 
            currentDir = new File(chosenFile.getParent()); // the file's directory

            /* The contents of the selected url file are read into a sequence.
               If the file doesn't contain only URLs or if the file can't be
               read from, the readToSeq method throws an exception.
            */
            URLFile urlFile = new URLFile(chosenFile);
            Sequence urlSeq = new SinglyLinkedList(); 
            urlFile.readToSeq(urlSeq); 
            isFileReadIn = true;

            /* The sequence of URLs forms the basis of a menu. The user
               can select a URL from the menu and its source will be
               downloaded for viewing.
             */
            URLMenu urlMenu = new URLMenu(urlSeq, chosenFile);
            urlMenu.show();
            urlMenu.invokeChosenItem();
         }
         catch (NotAURLFileException e)
         {  System.out.println("The file you have chosen is not a file of URLs.");
            System.out.println("Check that each item in the file is a URL.");
         }
         catch (URLFileReadWriteException e)
         {  System.out.println("The file you have chosen cannot be opened for reading.");
            System.out.println("Check your file system.");
         }
      } while (! isFileReadIn);
   }
      
   /**
    * The browser application's main method. (It all starts here!)
    * This creates and launches a browser object.
    *
    * @param args in the zero-th element of this array, the user
    * should supply (at the command-line) the directory where
    * she wants the first file dialog to begin. The user must supply
    * exactly one such name and it must be an existing directory.
    */
   public static void main(String[] args) 
   {  CheapBrowser cb = new CheapBrowser();

      if (args.length == 1) 
      {  File dialogDir = new File(args[0]);
         if (! dialogDir.exists()) 
         {  System.out.println(args[0] + " does not exist.");
         } 
         else if (! dialogDir.isDirectory()) 
         {  System.out.println(args[0] + " is not a directory.");
         } 
         else 
         {  // Launch browser with user-supplied diretcory
            cb.launch(new File(dialogDir.getAbsolutePath()));
         }
      } 
      else // args.length != 1
      {  System.out.println("Usage: java CheapBrowser <directory>");
      }
   }
}