package consoleIO;

import datastructs.*;
import utilities.*;
import java.util.Enumeration;
import java.io.*;

/**
 * A class that represents simple menus.
 * ConsoleMenu objects are displayed on the console; user selections are
 * integers typed at the keyboard.
 * @author Derek Bridge
 */

public class ConsoleMenu
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new console menu object with the specified title
    * and prompt.
    *
    * @param theTitle the title of the menu; will be displayed above
    * the menu; can be null (in which case, no title is displayed).
    * @param thePrompt the text used to prompt the user; must be non-null
    * and non-empty.
    */
   public ConsoleMenu(final String theTitle, final String thePrompt)
   {  if (thePrompt == null || thePrompt.trim().equals(""))
      {  throw new PreconditionException(
            "ConsoleMenu Precondition Error: invalid menu prompt: " +
            thePrompt);
      }

      title = theTitle;
      items = new SinglyLinkedList();
      prompt = thePrompt;
      console = new ConsoleInput();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* ------Getters------------------------------------------------------- */

   /**
    * Displays the menu on the console.
    */
   public void show()
   {  showTitle();
      /* Display each of the menu items, with numbers starting from
         LOWEST_ITEM_NUM.
       */
      Enumeration e = items.elements();
      for (int i = 0; i < items.size(); i++)
      {  showItem(i + LOWEST_ITEM_NUM, 
            ((ConsoleMenuItem) e.nextElement()).getDesc());
      }
   }

   /**
    * Gets a menu choice from the user. The user is prompted repeatedly until
    * a legitimate choice is given. The user's choice is given as an item
    * number, and this is what is returned.
    *
    * @return the user's choice as a position in the sequence of items.
    */
   public int getChosenItemNum()
   {  int menuChoice = 0;
      boolean isNotGoodResponse = true;

      do
      {  System.out.print("\n" + prompt);
         try
         {  menuChoice = console.readInt(); // may throw IOException or NumberFormatException
            if ((menuChoice < LOWEST_ITEM_NUM) ||
               (menuChoice >= LOWEST_ITEM_NUM + items.size()))
            {  throw new NumberFormatException();
            }
            isNotGoodResponse = false;
         }
         catch (NumberFormatException e)
         {  System.out.println("\nPlease enter a number between " +
               LOWEST_ITEM_NUM + " and " + (LOWEST_ITEM_NUM + items.size() - 1));
         }
         catch (IOException e)
         {  System.out.println("There was a problem in reading your input.");
            System.out.println("Check your system configuration.");
         }
      } while (isNotGoodResponse);

      /* The number returned will be used to index into the sequence of items.
         Sequence positions run from 0 up. But menu items are numbered
         starting from LOWEST_ITEM_NUM (typically 1), so this
         must be subtracted from the user's choice.
       */
      return menuChoice - LOWEST_ITEM_NUM; 
   }

   /**
    * Gets a menu choice from the user. The user is prompted repeatedly until
    * a legitimate choice is given. The user's choice is given as an item
    * number, but what is returned by this method is the actual ConsoleMenuItem
    * object from the sequence of menu items.
    * 
    * @return the menu item that the user chooses.
    */
   public ConsoleMenuItem getChosenItem() 
   {  int itemPosn = getChosenItemNum();
      return (ConsoleMenuItem) items.elementAt(itemPosn);
   }

   /** 
    * Gets a menu choice from the user. The user is prompted repeatedly until
    * a legitimate choice is given. The user's choice is given as an item
    * number. This method then gets the corresponding ConsoleMenuItem
    * object from the sequence of menu items. The menu items object's
    * invoke() method is then executed so that the menu choice launches
    * some appropriate processing.
    */
   public void invokeChosenItem() 
   {  getChosenItem().invoke();
   }

/* ------Setters------------------------------------------------------- */

   /**
    * Add an item to the menu.
    *
    * @param anItem an item to be added to the menu; must be non-null.
    * Each item is an object of class ConsoleMenuItem, thus containing 
    * a display string and a Command object that says what to do when 
    * this item is selected.
    */
   public void addItem(final ConsoleMenuItem anItem)
   {  if (anItem == null)
      {  throw new PreconditionException(
            "ConsoleMenu.addItem Precondition Error: invalid menu item: " +
            anItem);
      }

      items.addLast(anItem);
   }      


/* =======================================================================
       HELPER METHODS
   =======================================================================
*/   

   /**
    * Displays an overlined and underlined menu title.
    */
   private void showTitle()
   {  if (title != null)
      {  drawHorizontalLine(title.length(), '-');
         System.out.println(title);
         drawHorizontalLine(title.length(), '-');
      }
   }

   /**
    * Draws a horizontal line containing the specified
    * number of the given character.
    *
    * @param n the length of the line to be drawn; must be >= 0;
    * this precondition is checked by client code.
    * @param ch the character to draw; must be a drawable character;
    * this precondition is the responsibility of client code.
    */
   private static void drawHorizontalLine(final int n, final char ch)
   {  for(int i = 0; i < n; i++)
      {  System.out.print(ch);
      }
      System.out.println();
   }

   /**
    * Displays an item within a menu.
    *
    * @param anItemNum this menu item's number when being displayed.
    * @param anItemDesc this menu item's textual descrption, for display;
    * must be non-null and non-empty; this precondition is the
    * responsibility of client code.
    */
   private void showItem(final int anItemNum, final String anItemDesc)
   {  System.out.println("   " + anItemNum + ".  " + anItemDesc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   
  
   private String title; // the menu's title.
   private Sequence items; // the sequence of ConsoleMenuItems
   private String prompt; // the text used to prompt the user to make a choice.
   private ConsoleInput console; // the console object that can deliver keyboard input.

   private static final int LOWEST_ITEM_NUM = 1; // the item number of the first menu item.

}
