/**
 * A class that represents threshold logic units (TLUs).
 * These are standalone TLUs, not for use in networks of TLUs.
 * @author Derek Bridge
 */
public class StandaloneTLU
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new TLU with the given threshold, number of
    * inputs and weights. (The length of the weight array should
    * equal theNumOfInputs.)
    * @param theThreshold the threshold.
    * @param theNumOfInputs the number of inputs (i.e. the number
    * of sensors supplying input to this TLU).
    * @param theWeights the weights.
    */
   public StandaloneTLU(double theThreshold, int theNumOfInputs,
      double[] theWeights)
   {  /* The threshold is converted to an extra weighted input.
       */
      numOfInputs = theNumOfInputs + 1;
      weights = new double[numOfInputs];
      weights[0] = - theThreshold;
      for (int i = 1; i < numOfInputs; i++)  
      {  weights[i] = theWeights[i - 1];
      }
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Computes the activation (output) of this TLU, given a vector of
    * of sensor values. 
    * @param s an array of sensor values (the input to this TLU).
    * (The length of this array is whatever value the user supplied as 
    * theNumOfInputs in the constructor. It should exclude the extra 
    * input that results from treating the threshold as a weighted 
    * input. This extra input is handled directly in the code.)
    */
   public double activate(double[] s)
   {  double in = 0.0;
      in += weights[0] * 1.0; // the extra input for the threshold line
      for (int i = 1; i < numOfInputs; i++)
      {  in += weights[i] * s[i - 1];
      }
      return g(in);
   }

   /**
    * Implements the activation function of the TLU. In this case,
    * it returns 1 if its parameter >= 1; otherwise 0.
    * The reason this is implemented as a separate method is to allow
    * it to be easily overridden in subclasses of TLU.
    * @param x the weighted sum of the TLU's inputs.
    */
   public double g(double x)
   {  return x >= 0 ? 1 : 0;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private int numOfInputs;
   private double[] weights;
}
