import java.util.*;

/**
 * A class that represents states in the 8-puzzle.
 * This class uses an iconic representation: a 3 x 3 array of ints.
 * @author Derek Bridge
 */
public class EightPuzzleState
   implements IState
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a state in the 8-puzzle.
    * @param theTiles a 3 x 3 array of ints to specify a configuration 
    * of tiles.
    * Cell 0,0 is the top left; cell 1,0 is the top middle; and so on.
    * The integers are 0 (for the empty space) and then 1-8.
    * @param theBlankRow the row the blank is in.
    * @param theBlankCol the column the blank is in.
    * (Keeping track of the position of the blank is handy.)
    */
   public EightPuzzleState(int[][] theTiles, int theBlankRow,
      int theBlankCol)
   {  tiles = theTiles;
      blankRow = theBlankRow;
      blankCol = theBlankCol;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Tests whether this state is a goal state.
    * We'll hard-code a goal state:
    *      1  2  3
    *      8     4
    *      7  6  5
    * @return true if this state is a goal state; false otherwise.
    */
   public boolean isGoal()
   {  // Brute-force implementation!
      return 
         tiles[0][0] == 1 && tiles[0][1] == 2 && tiles[0][2] == 3 &&
         tiles[1][0] == 8 && tiles[1][1] == 0 && tiles[1][2] == 4 &&
         tiles[2][0] == 7 && tiles[2][1] == 6 && tiles[2][2] == 5;
   }

   /**
    * Returns the heuristic value for this state, i.e. an estimate
    * of the cost of the cheapest path to a goal state.
    * The value I use here is the number of tiles out of place.
    * This is an underestimate of the path cost, but it probably
    * underestimates too much to be a really good heuristic.
    */
   public int getHeuristicValue()
   {  // Brute-force implementation!
      int numOfTilesOutOfPlace = 0;
      if (tiles[0][0] != 1) numOfTilesOutOfPlace++;
      if (tiles[0][1] != 2) numOfTilesOutOfPlace++;
      if (tiles[0][2] != 3) numOfTilesOutOfPlace++;
      if (tiles[1][0] != 8) numOfTilesOutOfPlace++;
      if (tiles[1][1] != 0) numOfTilesOutOfPlace++;
      if (tiles[1][2] != 4) numOfTilesOutOfPlace++;
      if (tiles[2][0] != 7) numOfTilesOutOfPlace++;
      if (tiles[2][1] != 6) numOfTilesOutOfPlace++;
      if (tiles[2][2] != 5) numOfTilesOutOfPlace++;
      return numOfTilesOutOfPlace;
   }

   /**
    * Generates the successor states of this state.
    * @return an iterator for walking through the set of successor
    * states.
    * The successor states must be of class SuccessorState.
    */
   public Iterator getSuccessors()
   {  List succStates = new ArrayList(3);
      // If not topmost, move blank up
      if (blankRow != 0)
      {  succStates.add(new SuccessorState(
            new EightPuzzleState(
               exchangeTiles(blankRow, blankCol, blankRow - 1, blankCol),
               blankRow - 1, blankCol),
            "Up", 1));
      }
      // If not leftmost, move blank left
      if (blankCol != 0)
      {  succStates.add(new SuccessorState(
            new EightPuzzleState(
               exchangeTiles(blankRow, blankCol, blankRow, blankCol - 1),
               blankRow, blankCol - 1),
            "Left", 1));
      }
      // If not rightmost, move blank right
      if (blankCol != 2)
      {  succStates.add(new SuccessorState(
            new EightPuzzleState(
               exchangeTiles(blankRow, blankCol, blankRow, blankCol + 1),
               blankRow, blankCol + 1),
            "Right", 1));
      }
      // If not bottommost, move blank down
      if (blankRow != 2)
      {  succStates.add(new SuccessorState(
            new EightPuzzleState(
               exchangeTiles(blankRow, blankCol, blankRow + 1, blankCol),
               blankRow + 1, blankCol),
            "Down", 1));
      }
      return succStates.iterator();
   }

/* --Common public interface------------------------------------------- */

   /**
    * Returns true iff the parameter is equal to this configuration.
    * @param anObject the object being compared to this one.
    * @return true if the configurations are equal; false otherwise.
    */
   public boolean equals(Object anObject)
   {  if (this == anObject)
      {  return true;
      }
      else if (anObject == null ||
         getClass() != anObject.getClass())
      {  return false;
      }
      EightPuzzleState other = (EightPuzzleState) anObject;
      for (int i = 0; i < 3; i++)
      {  for (int j = 0; j < 3; j++)
         {  if (tiles[i][j] != other.tiles[i][j])
            {  return false;
            }
         }
      }
      return true;
   }

   /**
    * Returns a hash code for this object.
    */
   public int hashCode()
   {  int hashCode = 0;
      for (int i = 0; i < 3; i++)
      {  for (int j = 0; j < 3; j++)
         {  hashCode += i * j * tiles[i][j];
         }
      }
      return hashCode;
   }

   /**
    * Returns a string representation of this state.
    */
   public String toString()
   {  StringBuffer sb = new StringBuffer("+---+\n");
      for (int i = 0; i < 3; i++)
      {  sb.append("|");
         for (int j = 0; j < 3; j++)
         {  sb.append(tiles[i][j] != 0 ? tiles[i][j] + "" : " ");
         }
         sb.append("|\n");
      }
      sb.append("+---+\n");
      return sb.toString();
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Copies the 3 x 3 array.
    */
   private int[][] copyTiles()
   {  int[][] copy = new int[3][3];
      for (int i = 0; i < 3; i++)
      {  for (int j = 0; j < 3; j++)
         {  copy[i][j] = tiles[i][j];
         }
      }
      return copy;
   }

   /**
    * Exchange two tiles.
    * @param srcRow the source row
    * @param srcCol the source column.
    * @param destRow the destaination row.
    * @param destCol the destination column.
    * @return the updated configuration.
    */
   private int[][] exchangeTiles(
      int srcRow, int srcCol, int destRow, int destCol)
   {  int[][] copy = copyTiles();
      int temp = copy[srcRow][srcCol];
      copy[srcRow][srcCol] = copy[destRow][destCol];
      copy[destRow][destCol] = temp;
      return copy; 
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private int[][] tiles;
   private int blankRow;
   private int blankCol;
}
