import java.util.*;

/**
 * A class that represents nodes in a search tree for state space search.
 * @author Derek Bridge
 */
public class Node
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new node in a search tree.
    * this constructor is used for the start node (root of the tree).
    * @param theState the state associated with this node.
    */
   public Node(IState theState)
   {  state = theState;
      action = "";
      pathCost = 0; 
      parent = null;
      children = new LinkedList();
      depth = 0;
   }

   /**
    * Allocates a new node in a search tree.
    * @param theState the state associated with this node.
    * @param theAction the name of the operator that generated this state.
    * @param theCost the cost of the operator that generated this state.
    * @param theParent the search node that is this node's parent in the
    * search tree.
    * @param theDepth the depth of this node in the search tree.
    */
   public Node(IState theState, String theAction, int theCost,
      Node theParent, int theDepth)
   {  state = theState;
      action = theAction;
      pathCost = theParent.getPathCost() + theCost; 
      parent = theParent;
      children = new LinkedList();
      depth = theDepth;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Expand this node, i.e. compute its children in the tree.
    */
   public List expand()
   {  depth++;
      int childDepth = depth;
      /* Get this node's state and compute its successors.
       */
      Iterator iter = state.getSuccessors();
      /* Create nodes in the search tree from the successor states.
       */
      while (iter.hasNext())
      {  SuccessorState succ = (SuccessorState) iter.next();
         children.add(new Node(succ.getState(), succ.getAction(),
            succ.getCost(), this, childDepth));
      }
      return children;
   }

   /**
    * Returns a path in the search tree from the root to this node.
    */
   public List getPathFromRoot()
   {  List path = new LinkedList();
      Node node = this;
      do
      {  path.add(0, node);
         node = node.parent;
      } while (node != null);
      return path;
   }

   /**
    * Returns the state associated with this node.
    */
   public IState getState()
   {  return state;
   }

   /**
    * Returns the name of the operator that generated this node/state.
    * (Will return empty string for root.)
    */
   public String getAction()
   {  return action;
   }

   /**
    * Returns the cost of the path that leads to this node.
    * (Will return 0 for root node.)
    */
   public int getPathCost()
   {  return pathCost;
   }

   /**
    * Returns the parent node of this node in the tree.
    * (Will return null for the root node.)
    */
   public Node getParent()
   {  return parent;
   }

   /**
    * Returns the children nodes of this node.
    * (Will return an empty list if there are none.)
    */
   public List getChildren()
   {  return children;
   }

   /**
    * Returns the depth of this node in the tree.   
    * (Will return 0 for root node.)
    */
   public int getDepth()
   {  return depth;
   }

/* --Common public interface------------------------------------------- */

   /**
    * Returns true iff the parameter is equal to this state.
    * @param anObject the object being compared to this one.
    * @return true if the nodes are equal (which we take to be true if
    * the states that label the nodes are equal); false otherwise.
    */
   public boolean equals(Object anObject)
   {  if (this == anObject)
      {  return true;
      }
      else if (anObject == null ||
         getClass() != anObject.getClass())
      {  return false;
      }
      Node other = (Node) anObject;
      return state.equals(other.state);
   }

   /**
    * Returns a hash code for this object.
    */
   public int hashCode()
   {  return state.hashCode();
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private IState state;
   private String action;
   private int pathCost;
   private Node parent;
   private List children;
   private int depth;
}
