/**
 * A class that represents threshold logic units (TLUs).
 * These are for use in the output layer of fully-connected,
 * two-layered, feedforward networks. They receive their
 * input from the hidden units.
 * @author Derek Bridge
 */
public class OutputLayerTLU
   extends TLU
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new TLU with the given number of inputs, hidden units
    * and weights. (The lengths of the hidden unit and weight arrays 
    * should equal theNumOfInputs.)
    * @param theNumOfInputs the number of inputs (i.e. the number
    * of hidden units supplying input to this TLU, including `extras').
    * @param theHiddenUnits an array of hidden units that supply
    * input to this TLU (including `extras').
    * @param theWeights the weights.
    */
   public OutputLayerTLU(int theNumOfInputs, 
      HiddenLayerTLU[] theHiddenUnits, double[] theWeights)
   {  super(theNumOfInputs, theWeights);
      hiddenUnits = theHiddenUnits;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Computes the activation (output) of this TLU, from the outputs
    * of the hidden layer TLUs that are connected to this TLU.
    * You'll note that the outputs of the hidden layer units and the
    * output value of this TLU are cached in the array called inputs
    * and the variable called output.
    */
   public double activate()
   {  double in = 0.0;
      for (int i = 0; i < numOfInputs; i++)
      {  inputs[i] = hiddenUnits[i].getOutput();
         in += weights[i] * inputs[i];
      }
      output = g(in);
      return output;
   }

   /**
    * Implements the activation function of the TLU. In this case,
    * it returns 1 if its parameter >= 1; otherwise 0.
    * The reason this is implemented as a separate method is to allow
    * it to be easily overridden in subclasses of TLU.
    * @param x the weighted sum of the TLU's inputs.
    */
   public double g(double x)
   {  return x >= 0 ? 1 : 0;
   }

   /**
    * Returns the weight on one of the input lines entering this TLU.
    * @param i an index to the input line. (The index i can range
    * from 0 to numOfInputs inclusive. Line 0 is the extra line
    * that replaces the threshold.)
    */
   public double getWeight(int i)
   {  return weights[i];
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Sets the weight on one of the input lines entering this TLU.
    * @param i an index to the input line.  (The index i can range
    * from 0 to numOfInputs inclusive. Line 0 is the extra line
    * that replaces the threshold.)
    * @param theWeight the new weight.
    */
   public void setWeight(int i, double theWeight)
   {  weights[i] = theWeight;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected HiddenLayerTLU[] hiddenUnits;
}
