/**
 * A class that represents threshold logic units (TLUs).
 * These are for use in the hidden layer of fully-connected,
 * two-layered, feedforward networks. They receive their
 * input directly from the input units (sensors).
 * Their activation function is a sigmoid function.
 * @author Derek Bridge
 */
public class HiddenLayerTLU
   extends TLU
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new TLU with the given number of inputs and weights. 
    * (The length of the weight array should equal theNumOfInputs.)
    * @param theNumOfInputs the number of inputs (i.e. the number
    * of sensors supplying input to this TLU, including `extras').
    * @param theWeights the weights.
    */
   public HiddenLayerTLU(int theNumOfInputs, double[] theWeights)
   {  super(theNumOfInputs, theWeights);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Computes the activation (output) of this TLU, given a vector of
    * of sensor values.
    * You'll note that the sensor values and output value are cached 
    * in the array called inputs and the variable called output.
    * As a way of improving the efficiency of the learning algorithm, 
    * it also computes and stores the derivative of the activation.
    * @param s an array of sensor values (the inputs to this TLU).
    * (The length of this array is whatever value the user supplied as 
    * theNumOfInputs in the constructor. It should exclude the extra 
    * input that results from treating the threshold as a weighted 
    * input. This extra input is handled directly in the code.)
    */
   public double activate(double[] s)
   {  double in = 0.0;
      inputs[0] = 1.0;
      in += weights[0] * inputs[0]; // the special input
      for (int i = 1; i < numOfInputs; i++)
      {  inputs[i] = s[i - 1];
         in += weights[i] * inputs[i];
      }
      output = g(in);
      gradient = derivg(in);
      return output;
   }

   /**
    * Implements the activation function of the TLU. In this case,
    * it's a sigmoid function.
    * @param x the weighted sum of the TLU's inputs.
    */
   public double g(double x)
   {  return  1.0 / (1 + Math.pow(Math.E, - x));
   }

   /**
    * Implements the derivative of the activation function of the TLU.
    * @param the weighted sum of the TLU's inputs.
    */
   public double derivg(double x)
   {  return output * (1 - output);
   }

   /**
    * Returns the weight on one of the input lines entering this TLU.
    * @param i an index to the input line. (The index i can range
    * from 0 to numOfInputs inclusive. Line 0 is the extra line
    * that replaces the threshold.)
    */
   public double getWeight(int i)
   {  return weights[i];
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Sets the weight on one of the input lines entering this TLU.
    * @param i an index to the input line.  (The index i can range
    * from 0 to numOfInputs inclusive. Line 0 is the extra line
    * that replaces the threshold.)
    * @param theWeight the new weight.
    */
   public void setWeight(int i, double theWeight)
   {  weights[i] = theWeight;
   }
}
