package world;
import java.util.*;

/**
 * A class that represents reflex agents that inhabit grid-like worlds
 * and that are implemented using production systems.
 * @author Derek Bridge
 */
public abstract class ProductionSystemAgent
   extends Agent
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new agent.
    */
   public ProductionSystemAgent()
   {  rules = createRules();
      /* All agents have a full set of light, smell and touch sensors
         mounted around their bodies. Readings from these sensors go into
         arrays. But, so that we can experiment with different kinds of
         agents, these sensors can be enabled/disabled. Whether they
         are enabled or not is specified by an array of booleans.
       */
      hasLightSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      lightReading = new float[World.NUM_OF_COMPASS_POINTS];
      hasSmellSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      smellReading = new float[World.NUM_OF_COMPASS_POINTS];
      smellTypeReading = new int[World.NUM_OF_COMPASS_POINTS];
      hasTouchSensor = new boolean[World.NUM_OF_COMPASS_POINTS];
      touchReading = new boolean[World.NUM_OF_COMPASS_POINTS];
      /* Readings from sensors are preprocessed to give a vector of
         propositions.
       */
      propositions = new HashMap();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */
  
   /**
    * Returns a string representation of this agent's rules.
    */
   public String toString()
   {  return rules.toString();
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Creates this agent's production system.
    * How this is done will be very agent-specific. So the definition
    * of this method is left to subclasses and here we just return
    * an empty list of rules.
    */
   public ProductionSystem createRules()
   {  return new ProductionSystem();
   }

   /**
    * What to do on each clock tick.
    * @param theTime an integer representing how many ticks have passed.
    */
   public void onTick(int theTime)
   {  sense();
      preprocess();
      StimulusResponseRule rule = rules.findFirstMatch(propositions);
      Action action = rule.getAction();
      action.execute(this);
   }

   /**
    * Take sensor readings.
    */
   public void sense()
   {  Cell neighbour = null;
      int sens = FRONT;
      int dir = direction;
      for (int i = 0; i < World.NUM_OF_COMPASS_POINTS; 
         i++, sens = (sens + 1) % World.NUM_OF_COMPASS_POINTS, 
         dir = World.spin(dir, 1))
      {  neighbour = cell.getNeighbour(dir);
         if (hasLightSensor[sens])
         {  lightReading[sens] = neighbour.getAmountOfLight();
         }
         if (hasSmellSensor[sens])
         {  smellReading[sens] = neighbour.getAmountOfSmell();
            smellTypeReading[sens] = neighbour.getSmellType();
         }
         if (hasTouchSensor[sens])
         {  touchReading[sens] = neighbour.someSpaceIsInUse();
         }
      }
   }

   /**
    * Convert the sensor readings into propositions.
    * How this is done will be very agent-specific. So the definition
    * of this method is left to subclasses.
    */
   public abstract void preprocess();

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected ProductionSystem rules;

   /* An agent can have a number of light/smell/touch sensors at various
      positions around its perimeter. The positions are give by named
      constants above, e.g. FRONT_OF_HEAD. 
      Arrays of booleans say whether they are enabled or not. 
      They must be explicitly enabled, and this would be done in subclasses.
      Their readings are kept in another array. Light and smell readings are 
      real numbers; but touch sensors simply return true or false;
    */
   protected boolean[] hasLightSensor;
   protected float[] lightReading;

   protected boolean[] hasSmellSensor;
   protected float[] smellReading;
   protected int[] smellTypeReading;

   protected boolean[] hasTouchSensor;
   protected boolean[] touchReading;

   /* In the  preprocessing stage, sensor readings are converted into
      propositions, whose names (Strings) and truth-values (Booleans)
      are recorded in this Map (hash table).
    */
   protected Map propositions;

   /* The following two objects are created here in class variables
      so that they don't need to be re-created repeatedly. These two
      can simply be re-used.
    */
   protected static final Boolean TRUE = new Boolean(true);
   protected static final Boolean FALSE = new Boolean(false);
}
