package world;
import java.io.*;
import java.util.*;
import java.awt.*;
import gui.*;

/**
 * A class that represents cells in grid-like worlds.
 * For efficiency, Cells and their graphical representations (and worlds
 * and their graphical representations) are collapsed into a single
 * class (whereas, for flexibility, PhysObjects and their graphical
 * representations are handled by separate classes).
 * @author Derek Bridge
 */
public class Cell
   implements Serializable
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new cell.
    * @param theWorld the world in which this cell is located.
    * @param x the x-coordinate of this cell.
    * @param y the y-coordinate of this cell.
    */
   public Cell(World theWorld, int x, int y)
   {  world = theWorld;
      this.x = x;
      this.y = y;
      neighbours = new Cell[World.NUM_OF_COMPASS_POINTS];
      contents = new LinkedList();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */
   
   /**
    * Returns this cell's neighbour in the specified direction.
    */
   public Cell getNeighbour(int theDirection)
   {  Cell neighbour = neighbours[theDirection];
      /* Now, a trick. If there is no neighbour (we're at the edge
         of the world), we return a dummy cell. This cell cannot
         be entered by any physical object, has no smell and no light.
         In general, rather than rely on this dummy cell, you'd be
         better off putting a wall of Bricks around the edge of any world
         you create.
       */
      if (neighbour == null)
      {  return OUTER_WORLD_CELL;
      }
      else 
      {  return neighbour;
      }
   }

   /**
    * Returns true iff this cell can accommodate another object.
    * @param anObject the object that wants the unused space in this cell.
    */
   public boolean canAccommodate(PhysObject anObject)
   {  if (contents.isEmpty())
      {  return true;
      }
      Iterator contentsIterator = contents.iterator();
      float spaceNeeded = anObject.getSpaceUsed();
      while (contentsIterator.hasNext())
      {  PhysObject object = (PhysObject) contentsIterator.next();
         spaceNeeded += object.getSpaceUsed();
         if (spaceNeeded > 1.0f)
         {  return false;
         }
      }
      return true;
   }

   /**
    * Returns true iff this cell is empty.
    */
   public boolean isEmpty()
   {  return contents.isEmpty();
   }

   /**
    * Returns true if some cell space is in use.
    */
   public boolean someSpaceIsInUse()
   {  if (contents.isEmpty())
      {  return false;
      }
      Iterator contentsIterator = contents.iterator();
      while (contentsIterator.hasNext())
      {  PhysObject object = (PhysObject) contentsIterator.next();
         if (object.getSpaceUsed() > 0.0f)
         {  return true;
         }
      }
      return false;
   }

   /**
    * Returns the amount of pen-and-ink that this cell is reeking of.
    */
   public float getAmountOfSmell()
   {  return smell;
   }

   /**
    * Returns the type of smell that this cell is reeking of (an int).
    * Clients are expected to use getAmountOfSmell() to check that the
    * amount of smell is non-zero before using this getter.
    */
   public int getSmellType()
   {  return smellType;
   }

   /**
    * Returns the amount of light that is illuminating this cell.
    */
   public float getAmountOfLight()
   {  return light;
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Sets a reference to a neighbouring cell.
    * @param theDirection the direction in which we find this neighbour.
    * @param theNeighbour the neighbouring cell.
    */
   public void setNeighbour(int theDirection, Cell theNeighbour)
   {  neighbours[theDirection] = theNeighbour;
   }

   /**
    * Adds an object to this cell.
    * We assume that client code makes sure that this cell can accommodate
    * this new object.
    * @param anObject the object being added (or null).
    */
   public void addObject(PhysObject anObject)
   {  contents.add(anObject);
   }

   /**
    * Removes an object from this cell.
    * @param theObject the object being removed.
    */
   public void removeObject(PhysObject theObject)
   {  contents.remove(theObject);
   }

   /**
    * Increase the quantity of a particular smell in the cell.
    * @param theSmellType an int that indicates which smell is
    * being left behind.
    * @param theSmellChange the change in the smell.
    */
   public void leaveSmell(int theSmellType, float theSmellChange)
   {  /* How does this work?
         For each compass point, it radiates out as far as is necessary
         to disseminate the new quantity of smell. (That's the main loop
         and the inner loop that tests for qty > 0.)
         But when moving N, E, S, or W, it also fills in adjacent cells
         to the left and right.
         Why so  complicated? Cells only know their immediate neighbours.
         Also, some neighbours will be `dummy' cells (off the screen).
         It's no good trying to get neighbours of `dummy' cells.
         So this algorithm avoids trying to naviagte from diagonals to
         their neighbours.
         As it radiates, the smell only effects cells where there is 
         already the same type of smell or where this is both  a new and
         stronger smell.
       */
      storeSmell(this, theSmellChange, theSmellType);
      float qty = 0.0f;
      Cell cell = null;
      Cell c = null;
      int d = 0;
      for (int dir = World.EAST; dir < World.NUM_OF_COMPASS_POINTS; dir++)
      {  cell = this;
         qty = theSmellChange;
         for (int i = 0; qty > 0; qty -= world.getSmellFallOffRate(theSmellType), i++)
         {  cell = cell.getNeighbour(dir);
            storeSmell(cell, qty, theSmellType);
            if (dir == World.NORTH || dir == World.EAST || 
               dir == World.SOUTH || dir == World.WEST)
            {  d = World.spin(dir, 2);
               c = cell;
               for (int j = 0; j < i; j++)
               {  c = c.getNeighbour(d);
                  storeSmell(cell, qty, theSmellType);
               }
               d = World.spin(dir, -2);
               c = cell;
               for (int j = 0; j < i; j++)
               {  c = c.getNeighbour(d);
                  storeSmell(cell, qty, theSmellType);
               }               
            }
         }
      }
   }

   /**
    * Increase the quantity of light in the cell.
    * @param theLightChange the change in the light.
    */
   public void shedLight(float theLightChange)
   {  /* How does this work?
         For each compass point, it radiates out as far as is necessary
         to disseminate the new quantity of light. (That's the main loop
         and the inner loop that tests for qty > 0.)
         But when moving N, E, S, or W, it also fills in adjacent cells
         to the left and right.
         Why so  complicated? Cells only know their immediate neighbours.
         Also, some neighbours will be `dummy' cells (off the screen).
         It's no good trying to get neighbours of `dummy' cells.
         So this algorithm avoids trying to naviagte from diagonals to
         their neighbours.
       */
      light += theLightChange;
      float qty = 0.0f;
      Cell cell = null;
      Cell c = null;
      int d = 0;
      for (int dir = World.EAST; dir < World.NUM_OF_COMPASS_POINTS; dir++)
      {  cell = this;
         qty = theLightChange;
         for (int i = 0; qty > 0; qty -= world.getLightFallOffRate(), i++)
         {  cell = cell.getNeighbour(dir);
            cell.light += qty;
            if (dir == World.NORTH || dir == World.EAST || 
               dir == World.SOUTH || dir == World.WEST)
            {  d = World.spin(dir, 2);
               c = cell;
               for (int j = 0; j < i; j++)
               {  c = c.getNeighbour(d);
                  c.light += qty;
               }
               d = World.spin(dir, -2);
               c = cell;
               for (int j = 0; j < i; j++)
               {  c = c.getNeighbour(d);
                  c.light += qty;
               }               
            }
         }
      }
   }

   /**
    * What to do on each clock tick.
    * @param theTime an integer representing how many ticks have passed.
    */
   public void onTick(int theTime)
   {  /* The light vanishes. (If a light source is still around it
         can shed more light.)
       */
      light = 0.0f;
      /* The smell decreases by a little bit. (But a passing agent might 
         give leave additional pong.)
       */
      if (smell > 0)
      {  smell -= world.getSmellEvaporationRate(smellType);
         if (smell < 0.0f)
         {  smell = 0.0f;
         }
      }
   }
   /**
    * Draws this component on the graphics context.
    */
   public void draw(Graphics2D g2d)
   {  if (world.gridLinesAreVisible())
      {  g2d.setColor(world.getGridLineColour());
         g2d.drawRect(x, y, 1, 1);
      }
      if (smell > 0.0f && world.smellIsVisible())
      {  g2d.setColor(world.getColourOfSmell(smellType));
         g2d.fillRect(x, y, 1, 1);
      }
      else if (light > 0.0f && world.lightIsVisible())
      {  g2d.setColor(world.getColourOfLight()); 
         g2d.fillRect(x, y, 1, 1);
      }
      else
      {  g2d.setColor(world.getBackgroundColour());
         g2d.fillRect(x, y, 1, 1);
      }
      if (! contents.isEmpty())
      {  if (contents.size() == 1)
         {  PhysObjectGraphic physObjectGraphic = 
               ((PhysObject)contents.get(0)).getPhysObjectGraphic();
            if (physObjectGraphic != null)
            {  physObjectGraphic.draw(g2d, x, y);
            }
         }
         else // deal with multiple objects
         {  // simple solution (can be rewritten): draw highest priority object only
            PhysObjectGraphic highestPriorityGraphic = null;
            int highestPriority = 0;
            Iterator cIter = contents.iterator();
            while (cIter.hasNext())
            {  PhysObjectGraphic physObjectGraphic = 
                  ((PhysObject)(cIter.next())).getPhysObjectGraphic();
               if (physObjectGraphic != null)
               {  int priority = physObjectGraphic.getDrawingPriority();
                  if (priority > highestPriority)
                  {  highestPriorityGraphic = physObjectGraphic;
                     highestPriority = priority;
                  }
               }
            }
            if (highestPriorityGraphic != null)
            {  highestPriorityGraphic.draw(g2d, x, y);
            }
         }
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Updates the variables that keep track of a cell's smell.
    * @param cell the cell whose variables are being updated.
    * @param amountOfSmell the amount of stink being deposited.
    * @param smellType the type of stink being deposited.
    */
   private static void storeSmell(Cell cell, float amountOfSmell,
      int smellType)
   {  if (cell.smell == 0.0f)
      {  cell.smell = amountOfSmell;
         cell.smellType = smellType;
      }
      else // there is already a smell
      {  if (cell.smellType == smellType) // same type of smell
         {  cell.smell += amountOfSmell;
         }
         else // different cell
         {  if (cell.smell <= amountOfSmell) // new smell is stronger
            {  cell.smell = amountOfSmell;
               cell.smellType = smellType;
            }
            // else new smell is weaker, do nothing
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected World world;
   protected int x;
   protected int y;
   protected Cell[] neighbours;
   protected java.util.List contents;
   protected float smell;
   protected int smellType;
   protected float light;

   private static final Cell OUTER_WORLD_CELL = new Cell(null, -999, -999)
   {  public boolean canAccommodate(PhysObject anObject)
      {  return false;
      } 
      public float getAmountOfSmell()
      {  return 0.0f;
      }
      public int getSmellType()
      {  return 0;
      }
      public float getAmountOfLight()
      {  return 0.0f;
      }
      // No other methods should ever be called on this object.
   };
}
