import java.awt.*;
import javax.swing.*;

/**
 * A class that represents panels of a GUI for view/entry of countries.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class CountryPanel
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new panel for view/entry of countries.
    *
    * @param theDisplayOption an integer that specifies whether we want
    * to view countries or enter new countries.
    */
   public CountryPanel(int theDisplayOption)
   {  boolean state = (theDisplayOption == ENTRY);
      JLabel nameLbl = new JLabel("Name:");
      nameTF = new JTextField("", TF_COLS);
      nameTF.setEnabled(state);
      JLabel popLbl = new JLabel("Population:");
      popTF = new JTextField("", TF_COLS);
      popTF.setEnabled(state);
      JLabel currLbl = new JLabel("Currency:");
      currTF = new JTextField("", TF_COLS);
      currTF.setEnabled(state);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      setLayout(gbl);      
      gbc.insets = new Insets(5, 5, 5, 0); 
      gbc.fill = GridBagConstraints.HORIZONTAL; 
      gbc.anchor = GridBagConstraints.WEST;
      addComp(nameLbl, this, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), this, gbl, gbc, 0, 1, 1, 1, 0, 0);
      addComp(nameTF, this, gbl, gbc, 0, 2, 1, 1, 0, 0);
      addComp(popLbl, this, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), this, gbl, gbc, 1, 1, 1, 1, 0, 0);
      addComp(popTF, this, gbl, gbc, 1, 2, 1, 1, 0, 0);
      addComp(currLbl, this, gbl, gbc, 2, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), this, gbl, gbc, 2, 1, 1, 1, 0, 0);
      addComp(currTF, this, gbl, gbc, 2, 2, 1, 1, 0, 0);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Public class constants-------------------------------------------- */

   /**
    * A manifest value. IN the constructor, when theDisplayOption ==
    * ENTRY (i.e. 0), the user wants to insert countries into the file.
    */
   public static final int ENTRY = 0;

   /**
    * A manifest value. IN the constructor, when theDisplayOption ==
    * VIEW (i.e. 1), the user wants to retrieve countries from the file.
    */
   public static final int VIEW = 1;

   /**
    * The width of the textfields (in columns).
    */
   public static final int TF_COLS = 20;

/* --Getters----------------------------------------------------------- */

   /**
    * Get country details from the textfields.
    *
    * @return the Country object created from what's in the textfields.
    * @exception CountryDataInputException if the user's input is illegal.
    */
   public Country getCountry()
      throws CountryDataInputException
   {  try
      {  return new Country(
            nameTF.getText().trim(),
            Integer.parseInt(popTF.getText().trim()),
            currTF.getText().trim());
      }
      catch (NumberFormatException nfe)
      {  throw new CountryDataInputException();
      }
   }
 
/* --Setters----------------------------------------------------------- */

   /**
    * Clears the textfields.
    */
   public void clear()
   {  nameTF.setText("");
      popTF.setText("");
      currTF.setText("");
   }

   /**
    * Set the textfields from the specified country.
    *
    * @param theCountry the country used to set the textfields.
    */
   public void setCountry(Country theCountry)
   {  nameTF.setText(theCountry.getName());
      popTF.setText("" + theCountry.getPopulation());
      currTF.setText(theCountry.getCurrency());
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Utility method for adding components to a container using
    * grid bag layout constraints.
    *
    * @param comp the component being added.
    * @param cont the container we're adding to.
    * @param gbl the layout manager.
    * @param gbc the layout manager's constraints object.
    * @param row the row we're positioning in.
    * @param col the column we're positioning in.
    * @param numRows the number of rows this component is to span.
    * @param numCols the number of columns this component is to span.
    * @param weightx horizontal resize weight.
    * @param weighty vertical resize weight.
    */
   public static void addComp(Component comp, Container cont, 
      GridBagLayout gbl, GridBagConstraints gbc, int row, int col, 
      int numRows, int numCols, double weightx, double weighty)
   {  gbc.gridx = col;
      gbc.gridy = row;
      gbc.gridwidth = numCols;
      gbc.gridheight = numRows;
      gbc.weightx = weightx;
      gbc.weighty = weighty;

      cont.add(comp, gbc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * Textfield for country name.
    */
   private JTextField nameTF;

   /**
    * Textfield for country population.
    */
   private JTextField popTF;

   /**
    * Textfield for country currency.
    */
   private JTextField currTF;
}
