import java.io.*;

/**
 * A class that represents objects that manage a file of countries.
 * @author Derek Bridge
 */
public class CountryFile
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new object that manages the file of countries.
    *
    * @param theFile the country file.
    */
   public CountryFile(File theFile)
   {   file = theFile;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Opens the file, finds a country with the specified name in the file,
    * and returns it (if there is one).
    * (To keep things simple, I assume the file does not contain duplicate
    * country names.)
    *
    * @param theTargetName the country name we're searching for.
    * @return the country object with the specified name.
    * @exception NoSuchCountryException if the file contains no country
    * with the specified name.
    * @exception CountryFileIOException if the file can't be opened, read
    * or closed. (This includes if the file is in the wrong format.)
    */
   public Country retrieve(String theTargetName)
      throws NoSuchCountryException, CountryFileIOException
   {  BufferedReader br = null;
      try
      {  br = new BufferedReader(
            new FileReader(file));
         while (true)
         {  String str = br.readLine();
            if (str == null)
            {  throw new EOFException();
            }
            String name = str.substring(0, str.indexOf('|'));
            int pop = Integer.parseInt(
               str.substring(str.indexOf('|') + 1, str.lastIndexOf('|')));
            String curr = str.substring(str.lastIndexOf('|') + 1);
            if (name.equals(theTargetName))
            {  return new Country(name, pop, curr);
            }
         }
      }
      catch (EOFException eofe)
      {  throw new NoSuchCountryException();
      }
      catch (NumberFormatException nfe)
      {  throw new CountryFileIOException();
      }
      catch (IOException ioe)
      {  throw new CountryFileIOException();
      }
      finally
      {  try
         {  if (br != null)
            {  br.close();
            }
         }
         catch (IOException ioe)
         {  throw new CountryFileIOException();
         }
      }
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Opens the file, inserts the specified country.
    * (To keep things simple, I do not check whether this causes insertion
    * of a duplicate country.)
    *
    * @param theCountry the country to insert.
    * @exception CountryFileIOException if the file can't be opened, written
    * to or closed.
    */
   public void insert(Country theCountry)
      throws CountryFileIOException
   {  PrintWriter pw = null;
      try
      {  pw = new PrintWriter(
            new BufferedWriter(
            new FileWriter(file.getName(), true))); // opens to append
         pw.println(theCountry.getName() + "|" + 
            theCountry.getPopulation() + "|" +
            theCountry.getCurrency());
      }
      catch (IOException ioe)
      {  throw new CountryFileIOException();
      }
      finally
      {  if (pw != null)
         {  pw.close();
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private File file;
}
