import java.io.*;

/**
 * A class that represents objects that manage a file of countries.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class CountryFile
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new object that manages the file of countries.
    *
    * @param theFile the country file.
    */
   public CountryFile(File theFile)
   {   file = theFile;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Opens the file, finds a country with the specified name in the file,
    * and returns it (if there is one).
    * (To keep things simple, I assume the file does not contain duplicate
    * country names.)
    *
    * @param theTargetName the country name we're searching for.
    * @return the country object with the specified name.
    * @exception NoSuchCountryException if the file contains no country
    * with the specified name.
    * @exception CountryFileIOException if the file can't be opened, read
    * or closed. (This includes if the file is in the wrong format.)
    */
   public Country retrieve(String theTargetName)
      throws NoSuchCountryException, CountryFileIOException
   {  DataInputStream dis = null;
      try
      {  dis = new DataInputStream(new BufferedInputStream(
            new FileInputStream(file)));
         while (true)
         {  String name = dis.readUTF();
            int pop = dis.readInt();
            String curr = dis.readUTF();
            if (name.equals(theTargetName))
            {  return new Country(name, pop, curr);
            }
         }
      }
      catch (EOFException eofe)
      {  throw new NoSuchCountryException();
      }
      catch (IOException ioe)
      {  throw new CountryFileIOException();
      }
      finally
      {  try
         {  if (dis != null)
            {  dis.close();
            }
         }
         catch (IOException ioe)
         {  throw new CountryFileIOException();
         }
      }
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Opens the file, inserts the specified country.
    * (To keep things simple, I do not check whether this causes insertion
    * of a duplicate country.)
    *
    * @param theCountry the country to insert.
    * @exception CountryFileIOException if the file can't be opened, written
    * to or closed.
    */
   public void insert(Country theCountry)
      throws CountryFileIOException
   {  DataOutputStream dos = null;
      try
      {  dos = new DataOutputStream(new BufferedOutputStream(
            new FileOutputStream(file.getName(), true))); // opens to append
         dos.writeUTF(theCountry.getName());
         dos.writeInt(theCountry.getPopulation());
         dos.writeUTF(theCountry.getCurrency());
      }
      catch (IOException ioe)
      {  throw new CountryFileIOException();
      }
      finally
      {  try
         {  if (dos != null)
            {  dos.close();
            }
         }
         catch (IOException ioe)
         {  throw new CountryFileIOException();
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The file being read from/written to.
    */
   private File file;
}
