import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A class that represents pie chart objects (as JPanels) where a segments
 * can be selected using a setter and it will highlight itself.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class SelectablePieChart
   extends PieChart
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new selectable pie chart object with the specified 
    * percentages.
    *
    * @param thePercentages an array of percentages to be presented
    * as a pie chart; it is assumed that client code has ensured that
    * the sum of the values in the array is 100, and that no value
    * in the array is too small to show up.
    * @param the desired colours for each pie chart segment; it is
    * assumed that client code has ensured that this array is the
    * same length as the percentages array.
    * @param theRaisedSegment an index to the segment that is to be
    * displayed as raised; it is assumed that the client code has
    * ensured that this is between 0 (incl.) and the lenght of the
    * percentages array (excl.), or -1 if no segment is to be raised.
    */
   public SelectablePieChart(double[] thePercentages, Color[] theColours,
      int theRaisedSegment)
   {  super(thePercentages, theColours);
      raisedSegment = theRaisedSegment;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/
   /**
    * Changes which segment is to be raised.
    *
    * @param theRaisedSegment an index to the segment that is to be
    * displayed as raised; it is assumed that the client code has
    * ensured that this is between 0 (incl.) and the lenght of the
    * percentages array (excl.), or -1 if no segment is to be raised.
    */
   public void setRaisedSegment(int theRaisedSegment)
   {  raisedSegment = theRaisedSegment;
      repaint();
   }

   /**
    * Draws the pie chart.
    *
    * g the Graphics context
    */
   public void paintComponent(Graphics g)
   {  /* Essential for proper background painting.
      */
      super.paintComponent(g);

      /* The superclass has taken care of painting the chart.
         All we have to do now is raise one of the segments.
      */
      if (raisedSegment >= 0)
      {  Color shadedColour = colours[raisedSegment];
         for (int i = 1; i < RAISE_AMOUNT; i++) 
         {  shadedColour = darken(shadedColour);
            g.setColor(shadedColour);
            g.fillArc(x, y - i, chartWidth, chartHeight,
               startAngles[raisedSegment], segmentAngles[raisedSegment]);
         }
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Darkens a specified colour.
    *
    * @param theColor the colour to be darkened.
    */
   private Color darken(Color theColor)
   {  return new Color((int) (theColor.getRed() * SHADE),  
         (int) (theColor.getGreen() * SHADE), 
         (int) (theColor.getBlue() * SHADE));
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * An index to the segment of the chart that is to be raised.
    */
   private int raisedSegment;

   /**
    * The raised segment is a darker colour than it was previously.
    * This number is used to darken the colour.
    */
   private static final double SHADE = 0.99;

   /**
    * The raise amount is how high the raised segment will grow.
    * In a more professional program, we would make sure that
    * this isn't going to make the raised segment disappear
    * behind the panel borders.
    */
   private static final int RAISE_AMOUNT = 12;

/* =======================================================================
       Test-DRIVER
   =======================================================================
*/
   public static void main(String[] args)
   {  JFrame frame = new JFrame(
         "A Selectable Pie Chart Demo: Winter Wombat Deaths");
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      Container contentPane = frame.getContentPane();
      double[] winterWombatDeaths = {10, 20, 30, 40};
      Color[] cols = {Color.red, Color.green, Color.cyan, Color.yellow};
      PieChart chart = new SelectablePieChart(winterWombatDeaths, cols, 3);
      contentPane.add(chart);
      frame.pack();
      frame.setVisible(true);
   }
}
