import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A class that represents pie chart objects (subclasses of JPanel).
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class PieChart
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new pie chart object with the specified percentages.
    *
    * @param thePercentages an array of percentages to be presented
    * as a pie chart; it is assumed that client code has ensured that
    * the sum of the values in the array is 100, and that no value
    * in the array is too small to show up.
    * @param the desired colours for each pie chart segment; it is
    * assumed that client code has ensured that this array is the
    * same length as the percentages array.
    */
   public PieChart(double[] thePercentages, Color[] theColours)
   {  super();

      /* Compute the start angles (how far round (anticlockwise) from 
         0 degrees, bearing in mind that 0 degrees is at 3 o' clock)
         and segment angles (the angle of the span of a segment).
      */
      percentages = thePercentages;
      colours = theColours;
      noOfSegments = percentages.length;
      startAngles = new int[noOfSegments];
      segmentAngles = new int[noOfSegments];
      int segmentStart = 0; // start at 0 degrees, 3 o' clock
      int i;
      for (i = 0; i < noOfSegments; i++)
      {  startAngles[i] = segmentStart;
         segmentAngles[i] = (int) Math.round(360 * percentages[i] / 100);
         segmentStart += segmentAngles[i];
         /* Compensate for (rounded) percentages exceeding a full circle
            by ignoring any excess!
         */
         if (segmentStart >= 360)
         {  segmentAngles[i] -= (segmentStart - 360);
            segmentStart = 360;
         }
      }
      /* Compensate for failing to complete the circle by adding to
         the last segment!
      */
      i--;
      if (startAngles[i] + segmentAngles[i] < 360)
      {  segmentAngles[i] += (360 - segmentAngles[i]);
      }
   }


/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Draws the pie chart.
    *
    * g the Graphics context
    */
   public void paintComponent(Graphics g)
   {  /* Essential for proper background painting.
      */
      super.paintComponent(g);

      /* The width and height available for painting is the panel
         width/height less the border width/heights.
      */
      int panelWidth = getSize().width;
      int panelHeight = getSize().height;
      Insets insets = getInsets(); // borders
      int availWidth = panelWidth - insets.left - insets.right;
      int availHeight = panelHeight - insets.top - insets.bottom;
      /* So we'll draw  our pie chart as an oval in a rectangle that
         sits within the available drawing area.
      */
      chartWidth = (int) (availWidth * WIDTH_SCALER);
      chartHeight = (int) (availHeight * HEIGHT_SCALER);
      /* The top left corner of the chart.
      */
      x = insets.left + ((availWidth - chartWidth) / 2);
      y = insets.top + ((availHeight - chartHeight) / 2);
      /* Now draw the chart: a series of arcs.
      */
      for (int i = 0; i < noOfSegments; i++)
      {  g.setColor(colours[i]);
         g.fillArc(x, y, chartWidth, chartHeight,
            startAngles[i], segmentAngles[i]);
      }
   }

   /**
    * Gets the preferred size of this pie chart.
    *
    * @return the preferred size as a Dimension object
    */
   public Dimension getPreferredSize()
   {  /* The preferred size is the panel's borders plus some
         default width and height.
      */
      Insets insets = getInsets();
      return new Dimension(DEFAULT_WIDTH + insets.left + insets.right,
         DEFAULT_HEIGHT + insets.top + insets.bottom);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * Array of percentages to be shown as a pie chart.
    */
   protected double[] percentages;

   /**
    * Array of colours to be used when drawing the pie chart (one for
    * each percentage).
    */
   protected Color[] colours;

   /**
    * No of segments in the chart (lengths of the arrays).
    */
   protected int noOfSegments;

   /**
    * Array of start angles for when drawing segments.
    */
   protected int[] startAngles;

   /**
    * Array of segment angles for when drawing segments.
    */
   protected int[] segmentAngles;

   /**
    * Top-left of the chart, x-axis.
    */
   protected int x;

   /**
    * Top-left of the chart, y-axis.
    */
   protected int y;

   /**
    * Width of the chart within the available width.
    */
   protected int chartWidth;

   /**
    * Height of the chart within the availabel height.
    */
   protected int chartHeight;

   /**
    * We arrive at chart width by scaling available width by this amount.
    */
   private static final double WIDTH_SCALER = 0.8;

   /**
    * We arrive at chart height by scaling available height by this amount.
    */
   private static final double HEIGHT_SCALER = 0.618;

   /**
    * A default component width for use in preferred component size.
    */
   private static final int DEFAULT_WIDTH = 400;

   /**
    * A default component height for use in preferred component size.
    */
   private static final int DEFAULT_HEIGHT = 300;

/* =======================================================================
       Test-DRIVER
   =======================================================================
*/
   public static void main(String[] args)
   {  JFrame frame = new JFrame("A Pie Chart Demo: Winter Wombat Deaths");
      frame.addWindowListener(new WindowAdapter()
      {  public void windowClosing(WindowEvent we)
         {  System.exit(0);
         }
      });
      Container contentPane = frame.getContentPane();
      double[] winterWombatDeaths = {10, 20, 30, 40};
      Color[] cols = {Color.red, Color.green, Color.cyan, Color.yellow};
      PieChart chart = new PieChart(winterWombatDeaths, cols);
      contentPane.add(chart);
      frame.pack();
      frame.setVisible(true);
   }
}
