import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;

/**
 * A class that represents burger bar GUIs, i.e. gui objects that allow
 * staff to take and process customer orders.
 * @author Derek Bridge 666 d.bridge@cs.ucc.ie
 */
public class BurgerBarGUI
   extends JPanel
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a panel containing components for ordering
    * dishes.
    *
    * @param theBurgerBar the burger bar this GUI pertains to
    */
   public BurgerBarGUI(BurgerBar theBurgerBar)
   {  super();

      /* Store the burger bar.
       */
      burgerBar = theBurgerBar;

      /* Create the four main parts of the GUI
       */
      JPanel logoP = createLogoPanel();
      JPanel bofP = createBOFPanel();
      JScrollPane orderP = createOrderPane();
      JPanel actionP = createActionPanel();

      /* Layout the 4 parts.
       */
      setBackground(BACK_COLOUR);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      setLayout(gbl);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.insets = new Insets(20, 20, 20, 20);
      addComp(logoP, this, gbl, gbc, 0, 0, 1, 3, 1, 1);
      addComp(bofP, this, gbl, gbc, 1, 0, 1, 1, 1, 1);
      addComp(orderP, this, gbl, gbc, 1, 1, 1, 1, 1, 1);
      addComp(actionP, this, gbl, gbc, 1, 2, 1, 1, 1, 1);
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Creates a panel that contains a logo.
    */
   public JPanel createLogoPanel()
   {  JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      JLabel logoLbl = new JLabel(LOGO);
      Font font = LOGO_FONT;
      logoLbl.setFont(font);
      logoLbl.setForeground(FORE_COLOUR);
      ImageIcon image1 = new ImageIcon(IMAGE1_FILENAME);
      JLabel image1Lbl = new JLabel(image1);
      image1Lbl.setBorder(BorderFactory.createRaisedBevelBorder());
      ImageIcon image2 = new ImageIcon(IMAGE2_FILENAME);
      JLabel image2Lbl = new JLabel(image2);
      image2Lbl.setBorder(BorderFactory.createCompoundBorder(
         BorderFactory.createRaisedBevelBorder(),
         BorderFactory.createEtchedBorder(BORDER_COLOUR, FORE_COLOUR)));
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.insets = new Insets(20, 20, 20, 20);
      addComp(image1Lbl, p, gbl, gbc, 0, 0, 1, 1, 1, 1);
      addComp(logoLbl, p, gbl, gbc, 0, 1, 1, 1, 1, 1);
      addComp(image2Lbl, p, gbl, gbc, 0, 2, 1, 1, 1, 1);
      return p;
   }

   /**
    * Creates a panel with buttons for choosing meals (one button for
    * each item in the burger bar's bill-of-fare).
    */
   private JPanel createBOFPanel()
   {  /* Create the buttons for each item in the bill of fare.
         Lay them out by type, e.g. all burgers in one row.
       */
      JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;

      java.util.List billOfFare = burgerBar.getBillOfFare();
      for (int i = 0; i < billOfFare.size(); i++)
      {  java.util.List dishes = (java.util.List) billOfFare.get(i);
         for (int j = 0; j < dishes.size(); j++)
         {  AbstractDish dish = (AbstractDish) dishes.get(j);
            JButton b = new JButton(dish.getName());
            b.setBackground(BACK_COLOUR);
            b.setForeground(FORE_COLOUR);
            addComp(b, p, gbl, gbc, i, j, 1, 1, 1, 1);
            b.addActionListener(new AddItemToOrderListener(dish, burgerBar));
         }
      }
      p.setBorder(BorderFactory.createTitledBorder(
         BorderFactory.createLineBorder(BORDER_COLOUR), BILL_OF_FARE));

      return p;
   }

   private JScrollPane createOrderPane()
   {  /* Create the list for displaying the customer's order.
       */
      list = new JList(burgerBar.getCurrentOrder());
      list.setForeground(FORE_COLOUR);
      list.setSelectionBackground(FORE_COLOUR);
      JScrollPane sp = new JScrollPane(list);
      list.setVisibleRowCount(5); 
      list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      return sp;
   }

   /**
    * Creates a panel with a button that offers the actions of
    * deleting from the order and moving to the next customer.
    */
   private JPanel createActionPanel()
   {  JPanel p = new JPanel();
      p.setBackground(BACK_COLOUR);
      JButton deleteB = new JButton(DELETE_ITEM);
      deleteB.setBackground(BACK_COLOUR);
      deleteB.setForeground(FORE_COLOUR);
      deleteB.setToolTipText(DELETE_ITEM_TIP);
      deleteB.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  int index = list.getSelectedIndex();
            if (index != -1)
            {  burgerBar.removeItemFromOrder(index);
            }
         }
      });

      JButton nextB = new JButton(NEXT_CUST);
      nextB.setBackground(BACK_COLOUR);
      nextB.setForeground(FORE_COLOUR);
      nextB.setToolTipText(NEXT_CUST_TIP);
      nextB.addActionListener(new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  burgerBar.concludeOrder();
         }
      });

      p.setLayout(new GridLayout(2, 1, 10, 10));
      p.add(deleteB);
      p.add(nextB);
      return p;
   }

   /**
    * Utility method for adding components to a container using
    * grid bag layout constraints.
    *
    * @param comp the component being added.
    * @param cont the container we're adding to.
    * @param gbl the layout manager.
    * @param gbc the layout manager's constraints object.
    * @param row the row we're positioning in.
    * @param col the column we're positioning in.
    * @param numRows the number of rows this component is to span.
    * @param numCols the number of columns this component is to span.
    * @param weightx horizontal resize weight.
    * @param weighty vertical resize weight.
    */
   private void addComp(Component comp, Container cont, GridBagLayout gbl,
      GridBagConstraints gbc, int row, int col, int numRows, int numCols,
      double weightx, double weighty)
   {  gbc.gridx = col;
      gbc.gridy = row;
      gbc.gridwidth = numCols;
      gbc.gridheight = numRows;
      gbc.weightx = weightx;
      gbc.weighty = weighty;

      cont.add(comp, gbc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   /**
    * The burger bar that this GUI pertains to.
    */
   private BurgerBar burgerBar;

   /**
    * The JList that displays the current customer's order.
    */
   private JList list;

   /**
    * The text that appears on the button used to conclude the current
    * customer's transaction and move on to the next customer.
    */
   private static final String NEXT_CUST = "Next Customer";

   /**
    * The text of the tool tip for the next button.
    */
   private static final String NEXT_CUST_TIP = 
      "Clears the current order in readiness for next customer";

   /**
    * The text that appears on the button that is used to delete
    * a selected item from the current customer's order.
    */
   private static final String DELETE_ITEM = "Delete Item";

   /**
    * The text of the tool tip for the delete button.
    */
   private static final String DELETE_ITEM_TIP =
      "Removes the selected item (if any) from the current order";

   /**
    * The wording that appears in the border of the panel containing
    * the bill of fare.
    */
   private static final String BILL_OF_FARE = 
      "Pig Out At Derek's Burger Bar";

   /**
    * The logo wording that appears as a banner across the top of
    * the GUI.
    */
   private static final String LOGO = "Derek's Burger Bar";

   /**
    * The font used for the logo.
    */
   private static final Font LOGO_FONT = new Font("Serif", Font.ITALIC, 36);

   /**
    * The name of the file in which the banner image is stored.
    */
   private static final String IMAGE1_FILENAME = "dog.gif";

   /**
    * The name of the file in which the banner image is stored.
    */
   private static final String IMAGE2_FILENAME = "singe.gif";

   /**
    * The background colour of this GUI.
    */
   private static final Color BACK_COLOUR = Color.white;

   /**
    * The foreground colour of this GUI.
    */
   private static final Color FORE_COLOUR = new Color(255, 0, 153);

   /**
    * The colour of borders in this GUI.
    */
   private static final Color BORDER_COLOUR = Color.blue;
}
