import java.util.*;

/**
 * A class that represents collections of CDs.
 * @author Derek Bridge
 */
public class CDCollection
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new, empty collection of CDs.
    */
   public CDCollection()
   {  cds = new Vector();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns all CDs in the collection.
    */
   public Vector findAll()
   {  return cds;
   }

   /**
    * Returns the CD in the collection with the specified 
    * identification number (if any). Although there can only be 
    * one such cd, a Vector is returned for consistency with
    * other search methods. It also means that with no match,
    * an empty Vector can be returned.
    *
    * @param theIdNum the identification number being sought.
    * @return the CD object with the specified id number or 
    * an empty Vector if there is no such CD.
    */
   public Vector findById(String theIdNum)
   {  Vector matchList = new Vector();
      boolean found = false;
      CD cd = null;
      Enumeration e = cds.elements();
      while (e.hasMoreElements() && ! found)
      {  cd = (CD) e.nextElement();
         found = cd.getIdNum().equals(theIdNum);
      }
      if (found)
      {  matchList.addElement(cd);
      }
      return matchList;
   }

   /**
    * Returns the cds in the collection whose artists are equal to or
    * are similar to the specified artist name.
    *
    * @param theArtist the artist being sought; must be non-null & non-empty.
    * @return an enumeration of the CDs in the colection whose
    * artists match the specified artist.
    */
   public Vector findByArtist(String theArtist)
   {  Vector matchList = new Vector();
      Enumeration e = cds.elements();
      while (e.hasMoreElements())
      {  CD cd = (CD) e.nextElement();
         if (StringUtility.matches(cd.getArtist(), theArtist))
         {  matchList.addElement(cd);
         }
      }
      return matchList;
   }

   /**
    * Returns the cds in the collection whose titles are equal to or
    * are similar to the specified title.
    *
    * @param theTitle the title being sought; must be non-null & non-empty.
    * @return an enumeration of the CDs in the colection whose
    * titles match the specified title.
    */
   public Vector findByTitle(String theTitle)
   {  Vector matchList = new Vector();
      Enumeration e = cds.elements();
      while (e.hasMoreElements())
      {  CD cd = (CD) e.nextElement();
         if (StringUtility.matches(cd.getTitle(), theTitle))
         {  matchList.addElement(cd);
         }
      }
      return matchList;
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Add the specified CD object to the collection.
    *
    * @param theCD the CD being added to the collection; must be 
    * non-null & not already in the collection.
    */
   public void add(CD theCD)
   {  cds.addElement(theCD);
   }

   /**
    * Remove the specified CD from the collection.
    *
    * @param theCD the CD to be removed; must be non-null and
    * a CD already in the collection.
    */
   public void remove(CD theCD)
   {  cds.removeElement(theCD);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/

   private Vector cds;
}
