import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A class that represents one particular GUI for Till objects.
 * @author Derek Bridge
 */
public class TillGUI 
   extends JPanel
   implements ITillGUI
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new till GUI object.
    */
   public TillGUI(Till theTill)
   {  /* Store this GUIs till and till controller (listener)
       */
      till = theTill;
      tc = new TillController(till, this); 

      /* Create the four main parts of the GUI
       */
      JPanel custP = createCustPanel();
      JPanel numPadP = createNumPadPanel();
      JPanel actionP = createActionPanel();
      JPanel tillTotP = createTillTotPanel();

      /* Layout the 4 parts .
       */
      setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      setLayout(gbl);
      gbc.fill = GridBagConstraints.NONE;
      gbc.anchor = GridBagConstraints.CENTER;
      gbc.insets = new Insets(20, 0, 0, 0);
      addComp(custP, this, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(numPadP, this, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(actionP, this, gbl, gbc, 2, 0, 1, 1, 0, 0);
      addComp(tillTotP, this, gbl, gbc, 3, 0, 1, 1, 0, 0);

      /* Initialise the fields in the GUI (this is done by
         querying the till object).
       */
      update();
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Return the value in the price textfield.
    *
    * @return the value in the price textfield (as an int).
    */
   public int getPriceEntered()
   {  return Integer.parseInt(priceTF.getText());
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Place a value into the price textfiled.
    *
    * @param thePriceEntered the value to be placed into the price textfield
    * (an int).
    */
   public void setPriceEntered(int thePriceEntered)
   {  priceTF.setText("" + thePriceEntered);
   }

   /**
    * Bring the fields of the GUI up-to-date.
    * This is done by using getters on the till object and putting
    * these values into the fields of the GUI.
    */
   public void update()
   {  priceTF.setText("" + 0);
      owedTF.setText("" + till.getTotalOwed());
      floatTF.setText("" + till.getTillFloat());
      tillTotalTF.setText("" + till.getTillTotal());
      transTF.setText("" + till.getTransactionCount());
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Creates and lays out the components that deal with the current
    * customer: price of last item, and running total.
    */
   private JPanel createCustPanel()
   {  /* Create the fields and their labels.
       */
      priceLbl = new JLabel("Price:");
      priceTF = new JTextField(TF_WIDTH);
      priceTF.setEditable(false);
      priceTF.setBackground(Color.white);
      owedLbl = new JLabel("Total:");
      owedTF = new JTextField(TF_WIDTH);
      owedTF.setEditable(false);
      owedTF.setBackground(Color.white);

      /* Layout the fields and labels.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.WEST;
      gbc.insets = new Insets(4, 0, 0, 0);
      addComp(priceLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      addComp(priceTF, p, gbl, gbc, 0, 2, 1, 1, 0, 0);
      addComp(owedLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc,
         1, 1, 1, 1, 0, 0);
      addComp(owedTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
   
      return p;
   }

   /**
    * Creates and lays out the components that deal with the numeric
    * keypad.
    */
   private JPanel createNumPadPanel()
   {  /* Create the buttons of the numeric keypad.
       */
      digitB = new JButton[10];
      for (int i = 0; i < 10; i++)
      {  digitB[i] = new JButton("" + i);
         digitB[i].setBackground(Color.yellow);
      }
      enterB = new JButton(ENTER);
      enterB.setBackground(Color.cyan);

      /* Layout the buttons.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;
      addComp(digitB[0], p, gbl, gbc, 0, 0, 1, 3, 1, 1);
      for (int i = 1; i < 10; i++)
      {  addComp(digitB[i], p, gbl, gbc, (i + 2) / 3, 
            (i % 3) - 1, 1, 1, 1, 1);
      }
      addComp(enterB, p, gbl, gbc, 0, 3, 4, 1, 1, 1);

      /* Register a listener for the numeric buttons.
         They all share the same listener object, and it is defined
         as an anonymous class.
         Its handler simply updates the GUI's price field by appending
         the latest digit that the user has pressed onto the price so 
         far.
       */
      ActionListener lstnr = new ActionListener()
      {  public void actionPerformed(ActionEvent ae)
         {  String btnText = ae.getActionCommand();
            int digit = Integer.parseInt(btnText);
            int priceSoFar = getPriceEntered();
            int newPrice = priceSoFar * 10 + digit;
            setPriceEntered(newPrice);
         }
      };
      for (int i = 0; i < 10; i++)
      {  digitB[i].addActionListener(lstnr);
      }

      /* Register a listener for the Enter button.
       */
      enterB.addActionListener(tc);

      return p;
   }

   /**
    * Creates and lays out the components that deal with till actions:
    * submit and cancel.
    */
   private JPanel createActionPanel()
   {  /* Create the buttons that deal with the main till actions.
       */
      submitB = new JButton(SUBMIT);
      submitB.setBackground(Color.green);
      cancelB = new JButton(CANCEL);
      cancelB.setBackground(Color.red);

      /* Layout the buttons.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      p.setLayout(new BoxLayout(p, BoxLayout.X_AXIS));
      p.add(submitB);
      p.add(Box.createHorizontalStrut(4));
      p.add(cancelB);

      /* Register listeners for the Submit and Cancel buttons.
       */
      submitB.addActionListener(tc);
      cancelB.addActionListener(tc);

      return p;
   }

   /**
    * Creates and lays out the components that deal with the till totals:
    * initial float, total takings and number of transactions.
    */
   private JPanel createTillTotPanel()
   {  /* Create the textfields and their labels for showing the till
         information.
       */
      floatLbl = new JLabel("Float:");
      floatTF = new JTextField(TF_WIDTH);
      floatTF.setEditable(false);
      floatTF.setBackground(Color.white);
      tillTotalLbl = new JLabel("Till Total:");
      tillTotalTF = new JTextField(TF_WIDTH);
      tillTotalTF.setEditable(false);
      tillTotalTF.setBackground(Color.white);
      transLbl = new JLabel("Transactions:");
      transTF = new JTextField(TF_WIDTH);
      transTF.setEditable(false);
      transTF.setBackground(Color.white);

      /* Layout the textfield and their labels.
       */
      JPanel p = new JPanel();
      p.setBackground(Color.white);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      p.setLayout(gbl);
      gbc.fill = GridBagConstraints.HORIZONTAL;
      gbc.anchor = GridBagConstraints.WEST;
      gbc.insets = new Insets(4, 0, 0, 0);
      addComp(floatLbl, p, gbl, gbc, 0, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         0, 1, 1, 1, 0, 0);
      addComp(floatTF, p, gbl, gbc, 0, 2, 1, 1, 0, 0);
      addComp(tillTotalLbl, p, gbl, gbc, 1, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc,
         1, 1, 1, 1, 0, 0);
      addComp(tillTotalTF, p, gbl, gbc, 1, 2, 1, 1, 0, 0);
      addComp(transLbl, p, gbl, gbc, 2, 0, 1, 1, 0, 0);
      addComp(Box.createHorizontalStrut(10), p, gbl, gbc, 
         2, 1, 1, 1, 0, 0);
      addComp(transTF, p, gbl, gbc, 2, 2, 1, 1, 0, 0);
   
      return p;
   }

   /**
    * Utility method for adding components to a container using
    * grid bag layout constraints.
    *
    * @param comp the component being added.
    * @param cont the container we're adding to.
    * @param gbl the layout manager.
    * @param gbc the layout manager's constraints object.
    * @param row the row we're positioning in.
    * @param col the column we're positioning in.
    * @param numRows the number of rows this component is to span.
    * @param numCols the number of columns this component is to span.
    * @param weightx horizontal resize weight.
    * @param weighty horizontal resize weight.
    */
   private void addComp(Component comp, Container cont, GridBagLayout gbl,
      GridBagConstraints gbc, int row, int col, int numRows, int numCols,
      double weightx, double weighty)
   {  gbc.gridx = col;
      gbc.gridy = row;
      gbc.gridwidth = numCols;
      gbc.gridheight = numRows;
      gbc.weightx = weightx;
      gbc.weighty = weighty;

      cont.add(comp, gbc);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private Till till;
   private TillController tc;

   private JLabel priceLbl;
   private JTextField priceTF;
   private JLabel owedLbl;
   private JTextField owedTF;

   private JButton[] digitB;
   private JButton enterB;

   private JButton submitB;
   private JButton cancelB;

   private JLabel floatLbl;
   private JTextField floatTF;
   private JLabel tillTotalLbl;
   private JTextField tillTotalTF;
   private JLabel transLbl;
   private JTextField transTF;

   private static final int TF_WIDTH = 10;
}


