package web;

import java.net.*;
import java.io.*;
import java.util.Enumeration;
import datastructs.*;
import utilities.PreconditionException;

/**
 * Objects of this class represent files containing URLs.
 * Methods are provided for file-handling.
 * @author Derek Bridge
 */

public class URLFile
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new URL file object.
    *
    * @param theFile the file object that represnets the actual
    * file of URLs; must be non-null, must be a file, not a directory
    * and must exist.
    */
   public URLFile(final File theFile)
   {  if (theFile == null || theFile.isDirectory() || ! theFile.exists())
      {  throw new PreconditionException(
            "URLFile Constructor Precondition Error: invalid file: " +
            theFile);
      }

      file = theFile;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Reads each URL from the file and appends it to a sequence
    * of URLs. The sequence to append to is supplied as a parameter.
    *
    * @param theSeq the contents of the file are appended to this 
    * sequence; must be non-null.
    * @exception NotAURLFileException if what is being read from the
    * file is not a URL.
    * @exception URLFileReadWriteException if the file cannot be opened
    * for reading.
    */
   public void readToSeq(Sequence theSeq)
      throws NotAURLFileException, URLFileReadWriteException
   {  if (theSeq == null)
      {  throw new PreconditionException(
            "URLFile.readToSeq Precondition Error: invalid sequence: " +
            theSeq);
      }

      BufferedReader stream = null;
      try
      {  /* Open stream
          */
         stream = new BufferedReader(new FileReader(file));
         /* Read each String from the file and create a URL object from
            each. There may be IO exceptions and there may be problems
            parsing the strings as URLs.
          */
         boolean isEOF = false;
         do
         {  String nextString = stream.readLine(); 
            if (nextString == null)
            {  isEOF = true;
            }
            else
            {  theSeq.addLast(new URL(nextString));
            }
         } while (! isEOF);
         stream.close();
      }
      catch (MalformedURLException e)
      {  throw new NotAURLFileException();
      }
      catch (IOException e)
      {  throw new URLFileReadWriteException();
      }
   }

   /**
    * Writes each URL from the specified sequence into this file.
    * If the file does not exist, it will be created. If it does
    * exist, it will be overwritten.
    *
    * @param theSeq the sequence whose elements are written to the file; 
    * must be non-null; each item must be a URL.
    * @exception NotAURLFileException if what is being written is not
    * a URL.
    * @exception URLFileReadWriteException if the file cannot be opened
    * for writing.
    */
   public void writeFromSeq(final Sequence theSeq)
      throws NotAURLFileException, URLFileReadWriteException
   {  if (theSeq == null)
      {  throw new PreconditionException(
            "URLFile.readToSeq Precondition Error: invalid sequence: " +
            theSeq);
      }

      BufferedWriter stream = null;
      try
      {  /* Open stream
          */
         stream = new BufferedWriter(new FileWriter(file));
         /* Take each URL from the sequence and write its full path name
            to the file. There may be IOExceptions and there may be problems
            if the sequence contains objects other than URLs.
          */
         Enumeration enum = theSeq.elements();
         while (enum.hasMoreElements())
         {  URL url = (URL) enum.nextElement();
            stream.write(url.toString());
            stream.newLine(); // write line terminator
         }
         stream.close();
      }
      catch (ClassCastException e)
      {  throw new NotAURLFileException();
      }
      catch (IOException e)
      {  throw new URLFileReadWriteException();
      }
   }        

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   
  
   private File file; // the file object that represents this file of URLs.
}
