package consoleIO;

import java.io.*;

/**
 * A class that allows keyboard input, echoed in the console.
 * @author Derek Bridge
 */
public class ConsoleInput
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new console input object.
    */
   public ConsoleInput()
   {  reader = new BufferedReader(
         new InputStreamReader(System.in));
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Reads a line of text from the keyboard, returning it as a string.
    *
    * @return the string that the user entered.
    * @exception IOException if the input cannot be obtained.
    */
   public String readLine() throws IOException
   {  return reader.readLine();
   }

   /**
    * Reads a line of text from the keyboard, which should contain
    * an int; returns it as an int or throws an exception.
    *
    * @return the int that the user entered.
    * @exception IOException if the input cannot be obtained.
    * @exception NumberFormatException if the user's input cannot be
    * parsed as an int.
    */
   public int readInt() throws IOException, NumberFormatException
   {  return Integer.parseInt(reader.readLine().trim());
   }

   /**
    * Reads a line of text from the keyboard, which should contain
    * an int; if it does not contain an int, prompts the user
    * until s/he does enter an int.
    *
    * @return the int that the user (utlimately) enters.
    * @exception IOException if the input cannot be obtained.
    */
   public int readUntilInt() throws IOException
   {  int usersInt = 0;
      boolean isIntEntered = false;
      do
      {  try
         {  usersInt = Integer.parseInt(reader.readLine().trim());
            isIntEntered = true;
         }
         catch (NumberFormatException e)
         {  System.out.println("Your input number is not correct.");
            System.out.println("Your input must be a whole number " +
               "comprising only digits, e.g. 89");
            System.out.println("Please try to enter a whole number again:");
         }
      } while (! isIntEntered);
      return usersInt;
   }

   /**
    * Reads a line of text from the keyboard, which should contain
    * a double; returns it as a double or throws an exception.
    *
    * @return the double that the user entered.
    * @exception IOException if the input cannot be ontained.
    * @exception NumberFormatException if the user's input cannot be
    * parsed as a double.
    */
   public double readDouble() throws IOException, NumberFormatException
   {  return Double.valueOf(reader.readLine().trim()).doubleValue();
   }

   /**
    * Reads a line of text from the keyboard, which should contain
    * a double; if it does not contain a double, prompts the user
    * until s/he does enter a double.
    *
    * @return the double that the user (utlimately) enters.
    * @exception IOException if the input cannot be obtained.
    */
   public double readUntilDouble() throws IOException
   {  double usersDouble = 0.0;
      boolean isDoubleEntered = false;
      do
      {  try
         {  usersDouble = 
               Double.valueOf(reader.readLine().trim()).doubleValue();
            isDoubleEntered = true;
         }
         catch (NumberFormatException e)
         {  System.out.println("Your input number is not correct.");
            System.out.println("Your input must be a number either " +
               "with or without a decimal point, e.g. 89 or 56.75.");
            System.out.println("Please try to eneter a number again:");
         }
      } while (! isDoubleEntered);
      return usersDouble;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   

   /* All messages will be translated into operations on a buffered
      reader.
    */
   private BufferedReader reader;
}
