package consoleIO;

import java.io.*;
import utilities.*;

/**
 * A class that carries out file dialogs, i.e. it obtains
 * the name of a file from the user. The user's input comes
 * from the keyboard.
 * @author Derek Bridge
 */

public class ConsoleFileDialog
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/

   /**
    * Allocates a new console file dialog object.
    * This is an object for obtaining a file name from a user.
    * The name might be the name of an existing file or a new file,
    * as specified by the second parameter.
    *
    * @param theDialogTitle the string to display above the file
    * dialog menu (can be null, in which case no title is displayed).
    * @param int theMode either ConsoleFileDialog.LOAD or 
    * ConsoleFileDialog.SAVE, depending on whether this file is to be 
    * opened for loading (in which case it must exist) or saving (in 
    * which case it need not exist already); cannot be any other value.
    * @param theDirectory the absolute path name of the directory whose 
    * files will appear on the file dialog's menu; must be an existing 
    * directory; must be a directory and not a file.
    */
   public ConsoleFileDialog(final String theDialogTitle, 
      final int theMode, final File theDirectory)
   {  if (theMode != LOAD && theMode != SAVE)
      {  throw new PreconditionException(
            "ConsoleFileDialog Constructor Precondition Error: invalid " +
            "file open mode: " + theMode);
      }
      if (! theDirectory.isDirectory() || ! theDirectory.exists())
      {  throw new PreconditionException(
            "ConsoleFileDialog Constructor Precondition Error: invalid " +
            "directory: " + theDirectory);
      }
      title = theDialogTitle;
      mode = theMode;
      directory = theDirectory;

      /* A file dialog is just a menu. 
       */
      menu = new ConsoleMenu(title, PROMPT);

      /* The menu contains an entry for the current directory
         and the parent directory (if there is one).
       */
      addDirToDialog(".", directory);
      String parentDirName = directory.getParent();
      if (parentDirName != null)
      {  addDirToDialog("..", new File(parentDirName));
      }

      /* It also contains menu items for each of the files and
         directories that are in theDirectory.
       */
      String[] namesOfContentsOfDir = directory.list();
      if (namesOfContentsOfDir != null)
      {  for (int i = 0; i < namesOfContentsOfDir.length; i++)
         {  String name = namesOfContentsOfDir[i];
            File file = new File(directory, name);
            if (file.isDirectory())
            {  addDirToDialog(name, file);
            }
            else // it's a file
            {  addFileToDialog(name, file);
            }
         }
      }

      /* That's all the menu items if the mode is LOAD (you can only
         load from existing files).
         But, if the user wants to open a file in SAVE mode, there is
         another menu option: the option of creating a new file.
       */
      if (mode == SAVE)
      {  addCreateOptionToDialog();
      }
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* ------Public constants---------------------------------------------- */

   /**
    * Files can be opened in one of two modes: ConsoleFileDialog.LOAD
    *  or ConsoleFileDialog.SAVE
    */
   public static final int LOAD = 0; 
   public static final int SAVE = 1;

/* ------Getters------------------------------------------------------- */

   /**
    * Displays the menu on the console.
    */
   public void show()
   {  menu.show();
   }

   /**
    * Get the user to choose a file from the menu. 
    *
    * @return the user's chosen file name.
    */
   public File getUsersChoice() 
   {  /* The menu object can get the user's choice and execute the 
         command object associated with that choice. Ultimately, the 
         command object must produce a file, store it in chosenFile,
         from where it can be returned.
       */
      menu.invokeChosenItem();
      return chosenFile;
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/   

   /**
    * Adds a ConsoleMenuItem object to the file dialog menu for the
    * specified directory.
    * The object is created from a string (the description of the item)
    * and a Command object which says what to do when this item is
    * selected on the menu. The subclass of Command object needed
    * is FileDialogDirCommand.
    *
    * @param theDirName the directory name for this menu item; must be
    * non-null; this precondition is enforced by the client code.
    * @param theDir the corresponding File object for this name;
    * must be a directory and not a file; this precondition is enforced
    * by the client code.
    */
   private void addDirToDialog(final String theDirName, final File theDir)
   {  menu.addItem(new ConsoleMenuItem(theDirName,
         new FileDialogDirCommand(theDir)));
   }

   /**
    * Adds a ConsoleMenuItem object to the file dialog menu for the
    * specified file.
    * The object is created from a string (the description of the item)
    * and a Command object which says what to do when this item is
    * selected on the menu. The subclass of Command object needed
    * is FileDialogFileCommand.
    *
    * @param theFileName the file name for this menu item; must be
    * non-null; this precondition is enforced by the client code.
    * @param theFile the corresponding File object for this name;
    * must be a file and not a directory; this precondition is
    * enforced by the client.
    */
   private void addFileToDialog(final String theFileName, final File theFile)
   {  menu.addItem(new ConsoleMenuItem(theFileName,
         new FileDialogFileCommand(theFile)));
   }

   /**
    * Adds a ConsoleMenuItem object to the file dialog menu for the
    * option of creating a new file.
    * The object is created from a string (the description of the item)
    * and a Command object which says what to do when this item is
    * selected on the menu. The subclass of Command object needed
    * is FileDialogCreateCommand.
    *
    */
   private void addCreateOptionToDialog()
   {  menu.addItem(new ConsoleMenuItem(CREATION_TEXT,
         new FileDialogCreateCommand()));
   }

/* =======================================================================
       MEMBER CLASSES
   =======================================================================
*/  

   /** 
    * The class of command objects associated with directories in a
    * menu used in a file dialog.
    * When its execute() method is run, this begins a new file dialog
    * in which the user is invited to choose a file from the given directory.
    */
   private class FileDialogDirCommand
      extends Command
   {
      /*
       * Allocates this kind of Command object.
       *
       * @param theDir the directory for the new file dialog;
       * must be non-null and a directory and not a file; this 
       * precondition is enforced by the client code.
       */
      private FileDialogDirCommand(final File theDir)
      {  dir = theDir;
      }

      /*
       * The correct response to a user choosing a directory in
       * a file dialog is to display a new file dialog that invites
       * the user to choose a file from those within the chosen
       * directory.
       */
      public void execute()
      {  System.out.println();
         System.out.println();
         System.out.println("You have selected a directory.");
         System.out.println("Choose a file from within that directory");
         ConsoleFileDialog newDialog = new ConsoleFileDialog(title,
            mode, dir);
         newDialog.show();
         chosenFile = newDialog.getUsersChoice();
      }

      private File dir;
   }

   /** 
    * The class of command objects associated with files in a
    * menu used in a file dialog.
    * When its execute() method is run, simply the chosen file name
    * is stored so it can be obtained by a getter.
    */
   private class FileDialogFileCommand
      extends Command
   {
      /*
       * Allocates this kind of Command object.
       *
       * @param theFile the file the user can select;
       * must be non-null and a file and not a directory; this 
       * precondition is enforced by the client code.
       */
      private FileDialogFileCommand(final File theFile)
      {  file = theFile;
      }

      /*
       * The correct response to a user choosing a file in a file
       * dialog is to store the chosen file so that it can be
       * later retrieved by a getter.
       */
      public void execute()
      {  chosenFile = file;
      }

      private File file;
   }

   /** 
    * The class of command objects associated with creating new files in a
    * menu used in a file dialog.
    * When its execute() method is run, the user is asked for a new
    * file name; this is simply stored so it can be obtained by a getter.
    */
   private class FileDialogCreateCommand
      extends Command
   {  
      /*
       * Allocates this kind of Command object.
       */
      private FileDialogCreateCommand()
      {
      }

      /*
       * The correct response to a user choosing to create a new file
       * in a file dialog is to obtain the new filename from the user,
       * check that it is new and store the chosen file so that it can be
       * later retrieved by a getter.
       */
      public void execute() 
      {  boolean isLegitimateNewFileName = false;
         do
         {  try
            {  System.out.println();
               System.out.println();
               System.out.println(NEW_FILENAME_PROMPT); 
               String newFileName = new ConsoleInput().readLine().trim();
               /* There should be more code here to check that what
                  the user has supplied is a syntactically legitimate
                  filename, e.g. no spaces within it, etc.
                  I've omitted to save effort!
                */
               File newFile = new File(directory, newFileName);
               if (newFile.exists())
               {  System.out.println("You have entered the name of an already-existing file.");
                  System.out.println("Your new file cannot have the same name as an existing file.");
               }
               else
               {  chosenFile = newFile;
                  isLegitimateNewFileName = true;
               }
            }
            catch (IOException e)
            {  System.out.println("There was a problem in reading your input.");
               System.out.println("Check your system configuration.");
            }
         } while (! isLegitimateNewFileName);
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/   
  
   private ConsoleMenu menu; // the menu of file options.
   private String title; // the dialog title.
   private int mode; // the file mode: LOAD or SAVE.
   private File directory; // the directory this file dialog starts in.
   private File chosenFile; // the file the user eventually chooses.

   private static final String PROMPT = "Enter a number from the menu: ";
   private static final String CREATION_TEXT =
      "Choose to enter a new file name";
   private static final String NEW_FILENAME_PROMPT =
      "What do you want to call the new file: ";
}

