import java.util.*;

/**
 * A class for carrying out state space search.
 * @author Derek Bridge
 */
public class StateSpaceSearch
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new object that can carry out state space search.
    * @param theInitialState the initial state in this state space.
    */
   public StateSpaceSearch(IState theInitialState)
   {  initialState = theInitialState;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/
   /**
    * Carry out a breadth-first search.
    */
   public List breadthFirstSearch()
   {  agenda = new QueueAgenda();
      return getSolutionPath();
   }

   /**
    * Carry out a depth-first search.
    */
   public List depthFirstSearch()
   {  agenda = new StackAgenda();
      return getSolutionPath();
   }

   /**
    * Carry out a depth-bounded depth-first search.
    * @param theDepthLimit the depth limit of the search.
    */
   public List depthBoundedSearch(int theDepthLimit)
   {  agenda = new DepthBoundedAgenda(theDepthLimit);
      return getSolutionPath();
   }

   /**
    * Carry out an iterative-deepening depth-bounded depth-first search.
    */
   public List iterativeDeepeningSearch()
   {  List solutionPath = null;
      int depthBound = 0;
      do
      {  agenda = new DepthBoundedAgenda(depthBound);
         solutionPath = getSolutionPath();
         depthBound++;
      } while (solutionPath == null);
      return solutionPath;
   }

   /**
    * Carry out a least-cost search.
    */
   public List leastCostSearch()
   {  agenda = new PriorityQueueAgenda(new NodeCostComparator());
      return getSolutionPath();
   }

   /**
    * Carry out a greedy search.
    */
   public List greedySearch()
   {  agenda = new PriorityQueueAgenda(new NodeGreedyComparator());
      return getSolutionPath();
   }

   /**
    * Carry out an A* search.
    * (Strictly, this is only A* search if the heuristic underestimates
    * the cost of the cheapest path to the goal.)
    */
   public List aStarSearch()
   {  agenda = new PriorityQueueAgenda(new NodeAStarComparator());
      return getSolutionPath();
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   /**
    * Carry out a search; return the first solution path found;
    * return null if no path is found.
    */
   private List getSolutionPath()
   {  addNode(agenda, new Node(initialState));
      while (! agenda.isEmpty())
      {  Node currentNode = agenda.removeFrontNode();
         if (currentNode.getState().isGoal())
         {  return currentNode.getPathFromRoot();
         }
         addNodes(agenda, currentNode.expand());
      }
      return null;
   }

   /**
    * Adds a list of nodes to the agenda.
    * @param theAgenda the agenda we're adding to.
    * @param theNodes the nodes we're adding.
    */
   private void addNodes(IAgenda theAgenda, List theNodes)
   {  Iterator iter = theNodes.iterator();
      while (iter.hasNext())
      {  addNode(theAgenda, (Node) iter.next());
      }
   }

   /**
    * Adds a node to the agenda. This method is separated
    * out to allow it to be overridden in subclasses by 
    * alternative definitions (e.g. ones that avoid 
    * cycles in the state space).
    * @param theAgenda the agenda we're adding to.
    * @param theNode the node we're adding.
    */
   protected void addNode(IAgenda theAgenda, Node theNode)
   {  theAgenda.addNode(theNode);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private IState initialState;
   private IAgenda agenda;
}
