package gui;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import world.*;

/**
 * A class that represents dialogs for adding a brick to the world.
 * @author Derek Bridge
 */
public class BrickDialog
   extends JDialog
   implements ActionListener
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new dialog for adding a brick to the world.
    * @param theParent the parent component (this dialog will be centred
    * within its parent).
    * @param theWorld the world.
    */
   public BrickDialog(Component theParent, World theWorld)
   {  super();
      parent = theParent;
      world = theWorld;
      setTitle("Brick");
      JPanel dialogPanel = new JPanel();
      JLabel textLbl = new JLabel("Please set the co-ordinates");
      JLabel xLbl = new JLabel("x: ");
      xTextField = new JTextField("", 10);
      JLabel yLbl = new JLabel("y: ");
      yTextField = new JTextField("", 10);
      okBtn = new JButton("OK");
      okBtn.setActionCommand("ok");
      okBtn.addActionListener(this);
      cancelBtn = new JButton("Cancel");
      cancelBtn.setActionCommand("cancel");
      cancelBtn.addActionListener(this);
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      dialogPanel.setLayout(gbl);     
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;
      GuiUtility.addComp(textLbl, dialogPanel, gbl, gbc, 0, 0, 1, 4, 1, 1);
      GuiUtility.addComp(xLbl, dialogPanel, gbl, gbc, 1, 0, 1, 1, 1, 1);
      GuiUtility.addComp(xTextField, dialogPanel, gbl, gbc, 1, 1, 1, 1, 1, 1);
      GuiUtility.addComp(yLbl, dialogPanel, gbl, gbc, 1, 2, 1, 1, 1, 1);
      GuiUtility.addComp(yTextField, dialogPanel, gbl, gbc, 1, 3, 1, 1, 1, 1);
      GuiUtility.addComp(okBtn, dialogPanel, gbl, gbc, 2, 0, 1, 2, 1, 1);
      GuiUtility.addComp(cancelBtn, dialogPanel, gbl, gbc, 2, 2, 1, 2, 1, 1); 
      getContentPane().add(dialogPanel);
      pack();
   }

   /**
    * Makes this dialog visible, centred within its parent.
    * @param status whether to make it visible or invisible.
    */
   public void setVisible(boolean status)
   {  if (status)
      {  Point parentLocation = parent.getLocationOnScreen();
         Dimension parentSize = parent.getSize();
         Point point = new Point();
         Dimension size = getSize();  
         point.x = parentLocation.x + ((parentSize.width - size.width) / 2);
         point.y = parentLocation.y + ((parentSize.height - size.height) / 2);
         setLocation(point);
      }
      super.setVisible(status);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Responds to the user clicking either button in this dialog.
    */
   public void actionPerformed(ActionEvent ae)
   {  setVisible(false);
      String command = ae.getActionCommand();
      if (command.equals("ok"))
      {  try
         {  int x = Integer.parseInt(xTextField.getText().trim());
            int y = Integer.parseInt(yTextField.getText().trim());
            if (x < 0 || x >= world.getWorldWidth() || 
                y < 0 || y >= world.getWorldHeight())
            {  throw new NumberFormatException();
            }
            Brick brick = new Brick();
            Cell cell = world.getCells()[x][y];
            if (cell.canAccommodate(brick))
            {  brick.setCell(cell);
               cell.addObject(brick);
               world.addObject(brick);
               world.repaint();
            }
            else
            {  JOptionPane.showMessageDialog(parent, 
                  "The cell you specified has no room for this new object!",
                  "Cell occupied", JOptionPane.ERROR_MESSAGE);
            }
         }
         catch (NumberFormatException nfe)
         {  JOptionPane.showMessageDialog(parent, 
              "The x co-ordinate should be a whole number between 0 and " + 
              (world.getWorldWidth() - 1) +
              " inclusive\nThe y-cordinate should be a whole number between 0 and " +
              (world.getWorldHeight() - 1) + " inclusive.", 
              "Invalid number", JOptionPane.ERROR_MESSAGE);
         }
      }
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private Component parent;
   private World world;
   private JTextField xTextField;
   private JTextField yTextField;
   private JButton okBtn;
   private JButton cancelBtn;
}
