package gui;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import java.lang.reflect.*;
import world.*;

/**
 * A class that represents dialogs for adding agents to the world.
 * @author Derek Bridge
 */
public class AgentDialog
   extends JDialog
   implements ActionListener
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new dialog for adding agents to the world.
    * @param theParent the parent component (this dialog will be centred
    * within its parent).
    * @param theWorld the world.
    * @param theAgentClass a string specifying the name of agent's class.
    * @param theX the x-co-ordinate or -1 (undefined).
    * @param theY the y-co-ordinate or -1 (undefined).
    */
   public AgentDialog(Component theParent, World theWorld, String theAgentClass,
      int theX, int theY)
   {  super();
      parent = theParent;
      world = theWorld;
      agentClass = theAgentClass;
      setTitle(theAgentClass);
      x = theX;
      y = theY;
      JPanel dialogPanel = new JPanel();
      GridBagLayout gbl = new GridBagLayout();
      GridBagConstraints gbc = new GridBagConstraints();
      dialogPanel.setLayout(gbl);     
      gbc.fill = GridBagConstraints.BOTH;
      gbc.anchor = GridBagConstraints.CENTER;
      if (x == UNDEFINED)
      {  JLabel textLbl = new JLabel("Please set the co-ordinates & orientation");
         JLabel xLbl = new JLabel("x: ");
         xTextField = new JTextField("", 10);
         JLabel yLbl = new JLabel("y: ");
         yTextField = new JTextField("", 10);
         GuiUtility.addComp(textLbl, dialogPanel, gbl, gbc, 0, 0, 1, 4, 1, 1);
         GuiUtility.addComp(xLbl, dialogPanel, gbl, gbc, 1, 0, 1, 1, 1, 1);
         GuiUtility.addComp(xTextField, dialogPanel, gbl, gbc, 1, 1, 1, 1, 1, 1);
         GuiUtility.addComp(yLbl, dialogPanel, gbl, gbc, 1, 2, 1, 1, 1, 1);
         GuiUtility.addComp(yTextField, dialogPanel, gbl, gbc, 1, 3, 1, 1, 1, 1);
      }
      JLabel orientLbl = new JLabel("Orientation (0 = E, 1 = NE, 2 = N, 3 = NW, etc.):");
      orientTextField = new JTextField("", 10);
      okBtn = new JButton("OK");
      okBtn.setActionCommand("ok");
      okBtn.addActionListener(this);
      cancelBtn = new JButton("Cancel");
      cancelBtn.setActionCommand("cancel");
      cancelBtn.addActionListener(this);
      GuiUtility.addComp(orientLbl, dialogPanel, gbl, gbc, 2, 0, 1, 3, 1, 1);
      GuiUtility.addComp(orientTextField, dialogPanel, gbl, gbc, 2, 3, 1, 1, 1, 1);
      GuiUtility.addComp(okBtn, dialogPanel, gbl, gbc, 3, 0, 1, 2, 1, 1);
      GuiUtility.addComp(cancelBtn, dialogPanel, gbl, gbc, 3, 2, 1, 2, 1, 1); 
      getContentPane().add(dialogPanel);
      pack();
   }

   /**
    * Makes this dialog visible, centred within its parent.
    * @param status whether to make it visible or invisible.
    */
   public void setVisible(boolean status)
   {  if (status)
      {  Point parentLocation = parent.getLocationOnScreen();
         Dimension parentSize = parent.getSize();
         Point point = new Point();
         Dimension size = getSize();  
         point.x = parentLocation.x + ((parentSize.width - size.width) / 2);
         point.y = parentLocation.y + ((parentSize.height - size.height) / 2);
         setLocation(point);
      }
      super.setVisible(status);
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

   /**
    * Responds to the user clicking either button in this dialog.
    */
   public void actionPerformed(ActionEvent ae)
   {  setVisible(false);
      String command = ae.getActionCommand();
      if (command.equals("ok"))
      {  try
         {  if (x == UNDEFINED)
            {  x = Integer.parseInt(xTextField.getText().trim());
               y = Integer.parseInt(yTextField.getText().trim());
               orient = Integer.parseInt(orientTextField.getText().trim());
               if (x < 0 || x >= world.getWorldWidth() || 
                   y < 0 || y >= world.getWorldHeight() ||
                   orient < 0 || orient >= World.NUM_OF_COMPASS_POINTS)
               {  String msg =  
                    "The x co-ordinate should be a whole number between 0 and " + 
                    (world.getWorldWidth() - 1) +
                    " inclusive\nThe y-cordinate should be a whole number between 0 and " +
                    (world.getWorldHeight() - 1) + " inclusive.\n" +
                    "The orientation should be a whole number between 0 (east) and " +
                    "7 (southeast) inclusive";
                  throw new NumberFormatException(msg);
               }
            }
            else
            {  orient = Integer.parseInt(orientTextField.getText().trim());
               if (orient < 0 || orient >= World.NUM_OF_COMPASS_POINTS)
               {  String msg =
                    "The orientation should be a whole number between 0 (east) and " +
                    "7 (southeast) inclusive";
                  throw new NumberFormatException(msg);
               }
            }
            Agent agent = (Agent) Class.forName(agentClass).
               getConstructor(new Class[]{}).newInstance(new Object[]{});
            Cell cell = world.getCells()[x][y];
            if (cell.canAccommodate(agent))
            {  agent.setCell(cell);
               agent.setDirection(orient);
               cell.addObject(agent);
               world.addObject(agent);
               world.repaint();
            }
            else
            {  JOptionPane.showMessageDialog(parent, 
                  "The cell you specified has no room for this new object!",
                  "Cell occupied", JOptionPane.ERROR_MESSAGE);
            }
         }
         catch (NumberFormatException nfe)
         {  JOptionPane.showMessageDialog(parent, 
              nfe.getMessage(),
              "Invalid number", JOptionPane.ERROR_MESSAGE);
         }
         catch (ClassNotFoundException cnfe)
         {  reportConfigError();
         }
         catch (NoSuchMethodException nsme)
         {  reportConfigError();
         }
         catch (InstantiationException ie)
         {  reportConfigError();
         }
         catch (IllegalAccessException iae)
         {  reportConfigError();
         }
         catch (InvocationTargetException ite)
         {  reportConfigError();
         }
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/

   private void reportConfigError()
   {  JOptionPane.showMessageDialog(parent, 
         "This menu option has been misconfigured by the programmer.\n" +
         "Seek programmer assistance.", "System misconfigured", JOptionPane.ERROR_MESSAGE);
   }


/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private Component parent;
   private World world;
   private String agentClass;
   private int x = UNDEFINED;
   private int y = UNDEFINED;
   private int orient = UNDEFINED;
   private static final int UNDEFINED = -1;
   private JTextField xTextField;
   private JTextField yTextField;
   private JTextField orientTextField;
   private JButton okBtn;
   private JButton cancelBtn;
}
