package ga;
import java.util.*;

/**
 * A class that represents objects that can simulate evolution using
 * a genetic algorithm.
 * @author Derek Bridge
 */
public class GeneticAlgorithm
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new genetic algorithm.
    * @param theFactory a factory class that can supply the world and
    * agents we need.
    * @param thePopSize the size of population to use.
    * @param theTournamentSize the number of agents used in selection
    * tournaments. Clients must ensure this is smaller than the population
    * size.
    * @param theCombiningProportion the proportion [0,1] of each generation
    * that will be produced using combination on the previous generation.
    * @param theMutationProportion the proportion [0,1] of each generation
    * that will be produced using mutation on the previous generation.
    * @param theCopyingProportion the proportion [0,1] of each generation
    * that will be produced by copying from the previous population.
    * Clients are expected to check that these three propertions sum to 1.
    * (In fact, the code ignores the third of these three.)
    */
   public GeneticAlgorithm(IGAFactory theFactory, int thePopSize, 
      int theTournamentSize, float theCombiningProportion, 
      float theMutationProportion, float theCopyingProportion)
   {  factory = theFactory;
      popSize = thePopSize;
      pop = new IGAAgent[popSize];
      for (int i = 0; i < popSize; i++)
      {   pop[i] = factory.createAgent();
      }
      testWorld = factory.createWorld();
      // Evaluate this initial population.
      for (int i = 0; i < popSize; i++)
      {  testWorld.evaluateFitness(pop[i]);
      }
      tournamentSize = theTournamentSize;
      howManyFromCombining = (((int) (popSize * theCombiningProportion)) / 2) * 2;
      howManyFromMutating = (int) (popSize * theMutationProportion);
      howManyFromCopying = popSize - howManyFromCombining -
            howManyFromMutating;
   }

/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /**
    * Returns the best agent in the current population.
    */
   public IGAAgent getBestAgentInPopulation()
   {  IGAAgent bestAgentSoFar = null;
      float fitnessOfBestAgentSoFar = 0.0f;
      for (int i = 0; i < popSize; i++)
      {  if (pop[i].getFitness() > fitnessOfBestAgentSoFar)
         {  bestAgentSoFar = pop[i];
            fitnessOfBestAgentSoFar = bestAgentSoFar.getFitness();
         }
      }
      return bestAgentSoFar;
   }

   /**
    * Returns the whole population.
    */
   public IGAAgent[] getPopulation()
   {  return pop;
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Generates the next generation.
    */
   public void evolve()
   {  // Create new population by copying, combining and mutating.
      IGAAgent[] newPop = new IGAAgent[popSize];
      int i = 0;
      for (int j = 0; j < howManyFromCopying; j++, i++)
      {  newPop[i] = (IGAAgent) (tournamentSelection().clone());
      }
      for (int j = 0; j < howManyFromMutating; j++, i++)
      {  newPop[i] = tournamentSelection().createMutatedChild();
      }
      for (int j = 0; j < howManyFromCombining; j = j + 2, i = i + 2)
      {  List children = 
            tournamentSelection().createCombinedChildren(
               tournamentSelection());
         newPop[i] = (IGAAgent) children.get(0);
         newPop[i+1] = (IGAAgent) children.get(1);
      }
      pop = newPop;
      // Evaluate this new population.
      for (int k = 0; k < popSize; k++)
      {  testWorld.evaluateFitness(pop[k]);
      }
   }

   /**
    * Generates a fixed number of successive generations.
    * @param theNumGens the number of generations.
    */
   public void evolve(int theNumOfGens)
   {  for (int g = 0; g < theNumOfGens; g++)
      {  evolve();
      }
   }

   /**
    * Generates successive generations until a fitness threshold
    * is reached (if ever!).
    * @param theThreshold the fitness threshold.
    */
   public void evolve(float theThreshold)
   {  while (getBestAgentInPopulation().getFitness() < theThreshold)
      {  evolve();
      }
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Returns a selected agent from the population using tournament selection.
    */
   private IGAAgent tournamentSelection()
   {  IGAAgent bestAgentSoFar = pop[(int) (Math.random() * popSize)];
      for (int i = 1; i < tournamentSize; i++)
      {  int randPosn = (int) (Math.random() * popSize);
         IGAAgent candidateAgent = pop[randPosn];
         if (candidateAgent.getFitness() > bestAgentSoFar.getFitness())
         {  bestAgentSoFar = candidateAgent;
         }
      }
      return bestAgentSoFar;
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   private IGAFactory factory;
   private IGAAgent[] pop;
   private float[] fitnessScores;
   private int popSize;
   private int tournamentSize;
   private IGAWorld testWorld;
   private int howManyFromCombining;
   private int howManyFromMutating;
   private int howManyFromCopying;
}
