package demos;
import java.util.*;
import ga.*;
import world.*;

/**
 * A class that represents worlds in which evolved agents are tested on 
 * wall-following tasks.
 * @author Derek Bridge
 */
public class GAWalledWorld
   extends WalledWorld
   implements IGAWorld
{
/* =======================================================================
       CONSTRUCTORS
   =======================================================================
*/
   /**
    * Allocates a new world with given dimensions.
    * @param theWidth the width of the world.
    * @param theHeight the height of the world.
    * @param theScaleFactor the scale factor used for displaying the world.
    */
   public GAWalledWorld(int theWidth, int theHeight, int theScaleFactor)
   {  super(theWidth, theHeight, theScaleFactor);
      numOfTests = DEFAULT_NUM_OF_TESTS;
      testDuration = DEFAULT_TEST_DURATION;
   }
      
/* =======================================================================
       PUBLIC INTERFACE
   =======================================================================
*/

/* --Getters----------------------------------------------------------- */

   /** 
    * Evaluates the fitness of the given agent for `survival' in this
    * world. 
    * @param theAgent the agent being tested.
    */
   public void evaluateFitness(IGAAgent theAgent)
   {  Agent agent = (Agent) theAgent;
      float totalScore = 0.0f;
      for (int i = 0; i < numOfTests; i++)
      {  resetWorld();
         addAgent(agent);
         for (int t = 0; t < testDuration; t++)
         {  onTick(t);
         }
         totalScore += getTestScore();
         removeAgent(agent);
      }
      // the fitness is the average score for the different tests
      theAgent.setFitness(totalScore / numOfTests);
   }

/* --Setters----------------------------------------------------------- */

   /**
    * Sets the number of tests over which an agent will be evaluated.
    * @param theNumOfTests the number of tests.
    */
   public void setNumOfTests(int theNumOfTests)
   {  numOfTests = theNumOfTests;
   }

   /**
    * Sets the duration (in clock ticks) of the tests on which an agent will 
    * be evaluated.
    * @param theTestDuration the duration of the tests.
    */
   public void setTestDuration(int theTestDuration)
   {  testDuration = theTestDuration;
   }

/* =======================================================================
       HELPER METHODS
   =======================================================================
*/
   /**
    * Resets the world prior to evaluating an agent.  
    * In this world, we are using some CellMonitorAgents who see whether
    * cells get visited. Here, we reset these agents.
    */
   private void resetWorld()
   {  Iterator iter = monitors.iterator();
      while (iter.hasNext())
      {  CellMonitorAgent cm = (CellMonitorAgent) iter.next();
         cm.setWasVisited(false);
      }   
   }

   /**
    * Adds the agent to the world prior to evaluating it.  
    * Here agent placement is done randomly.
    * @param theAgent the agent to be evaluated.
    */
   private void addAgent(Agent theAgent)
   {  // keep trying different agent positions until a vacant one is found
      int x = 0;
      int y = 0;
      do
      {  x = (int) (Math.random() * width);
         y = (int) (Math.random() * height);
      } while (! cells[x][y].canAccommodate(theAgent));
      theAgent.setCell(cells[x][y]);
      cells[x][y].addObject(theAgent);
      objects.add(theAgent);
      // the agent orientation is also random (but I only E, N, W or S!)
      int randomDir = ((int) (Math.random() * NUM_OF_COMPASS_POINTS / 2) * 2);
      theAgent.setDirection(randomDir);
   }

   /** 
    * Returns a score for the agent on a test: the ratio of visited monitored
    * cells over the total number of monitored cells.
    */
   private float getTestScore()
   {  int visited = 0;
      Iterator iter = monitors.iterator();
      while (iter.hasNext())
      {  CellMonitorAgent cm = (CellMonitorAgent) iter.next();
         if (cm.wasVisited())
         {  visited++;
         }
      }
      return visited * 1.0f / monitors.size();
   }

   /**
    * Tidies up after an evaluation by removing the agent from the world.
    * @param theAgent the agent that has been evaluated.
    */
   private void removeAgent(Agent theAgent)
   {  theAgent.getCell().removeObject(theAgent);
      objects.remove(theAgent);
   }

/* =======================================================================
       INSTANCE VARIABLES & CLASS VARIABLES
   =======================================================================
*/
   protected List monitors;
   private int numOfTests;
   private int testDuration;

   private static final int DEFAULT_NUM_OF_TESTS = 3;
   private static final int DEFAULT_TEST_DURATION = 60;
}
